<?php

namespace Tests\Unit;

use App\Models\User;
use App\Models\Client;
use App\Models\Advocate;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class UserTest extends TestCase
{
    // Use the RefreshDatabase trait to ensure a fresh database state for each test
    use RefreshDatabase;

    /**
     * Test if a user can be created and exists in the database.
     */
    public function test_user_can_be_created()
    {
        // Create a user using the factory with specified attributes
        $user = User::factory()->create([
            'name' => 'Jane Doe',
            'email' => 'jane@example.com',
            'password' => bcrypt('secret'), // hash password before storing
        ]);

        // Assert that the user was successfully inserted into the users table
        $this->assertDatabaseHas('users', [
            'email' => 'jane@example.com',
        ]);
    }

    /**
     * Test the relationship: a user has one client.
     */
    public function test_user_has_one_client()
    {
        // Create a user
        $user = User::factory()->create();

        // Create a client associated with the user
        $client = Client::factory()->create([
            'user_id' => $user->id,
        ]);

        // Assert the relationship returns a Client instance
        $this->assertInstanceOf(Client::class, $user->client);

        // Assert the client returned belongs to the created user
        $this->assertEquals($client->id, $user->client->id);
    }

    /**
     * Test the relationship: a user has one advocate.
     */
    public function test_user_has_one_advocate()
    {
        // Create a user
        $user = User::factory()->create();

        // Create an advocate associated with the user
        $advocate = Advocate::factory()->create([
            'user_id' => $user->id,
        ]);

        // Assert the relationship returns an Advocate instance
        $this->assertInstanceOf(Advocate::class, $user->advocate);

        // Assert the advocate returned belongs to the created user
        $this->assertEquals($advocate->id, $user->advocate->id);
    }

    /**
     * (Optional) Test that the email_verified_at attribute is cast to a Carbon datetime instance.
     * Uncomment to use.
     */
    /*
    public function test_email_verified_at_is_cast_as_datetime()
    {
        $user = User::factory()->create([
            'email_verified_at' => now(),
        ]);

        // Assert that the casted attribute is an instance of Carbon
        $this->assertInstanceOf(\Illuminate\Support\Carbon::class, $user->email_verified_at);
    }
    */
}
