<?php

namespace Tests\Unit;

use App\Models\Client;
use App\Models\Quotation;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class QuotationTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_has_fillable_attributes()
    {
        $quotation = new Quotation();

        $this->assertEqualsCanonicalizing([
            'quote_number',
            'client_id',
            'type',
            'details',
            'amount',
            'quote_date',
            'status',
        ], $quotation->getFillable());
    }

    #[Test]
    public function it_belongs_to_a_client()
    {
        $client = Client::factory()->create();
        $quotation = Quotation::factory()->create([
            'client_id' => $client->id,
        ]);

        $this->assertInstanceOf(Client::class, $quotation->client);
        $this->assertEquals($client->id, $quotation->client->id);
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        $quotation = Quotation::factory()->create();
        $quotation->delete();

        $this->assertSoftDeleted('quotations', ['id' => $quotation->id]);
    }

    #[Test]
    public function it_auto_generates_quote_number_on_creation()
    {
        $quotation = Quotation::factory()->create();

        $this->assertNotNull($quotation->quote_number);
        $this->assertEquals('AYC/QT/' . str_pad($quotation->id, 3, '0', STR_PAD_LEFT), $quotation->quote_number);
    }

    #[Test]
    public function it_casts_quote_date_to_datetime()
    {
        $quotation = Quotation::factory()->create([
            'quote_date' => '2025-08-21',
        ]);

        $this->assertInstanceOf(\Illuminate\Support\Carbon::class, $quotation->quote_date);
        $this->assertEquals('2025-08-21', $quotation->quote_date->format('Y-m-d'));
    }

    #[Test]
    public function it_can_have_accepted_status()
    {
        $quotation = Quotation::factory()->accepted()->create();

        $this->assertEquals('accepted', $quotation->status);
    }

    #[Test]
    public function it_can_have_rejected_status()
    {
        $quotation = Quotation::factory()->rejected()->create();

        $this->assertEquals('rejected', $quotation->status);
    }
}