<?php

namespace Tests\Unit;

use App\Models\ApartmentDetail;
use App\Models\Client;
use App\Models\Document;
use App\Models\Expense;
use App\Models\Invoice;
use App\Models\Project;
use App\Models\ProjectClient;
use App\Models\ProjectExpense;
use App\Models\SubDivisionDetail;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;


class ProjectTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_project_and_generate_serial_number()
    {
        $project = Project::factory()->create();

        $this->assertNotNull($project->serial_number);
        $this->assertStringStartsWith('AYC/PJ/', $project->serial_number);

        $this->assertDatabaseHas('projects', [
            'id' => $project->id,
            'serial_number' => $project->serial_number,
        ]);
    }

    #[Test]
    public function it_has_fillable_attributes()
    {
        $project = new Project();

        $this->assertEqualsCanonicalizing([
            "name",
            "poc",
            "location",
            "description",
            "number_of_units",
            "type",
            "serial_number"
        ], $project->getFillable());
    }

    #[Test]
    public function it_can_have_clients_through_project_clients()
    {
        $project = Project::factory()->create();
        $client = Client::factory()->create();

        $project->clients()->attach($client->id, ['id' => 1]);

        $this->assertTrue($project->clients->contains($client));
    }

    #[Test]
    public function it_can_have_documents()
    {
        $project = Project::factory()->create();
        $document = Document::factory()->create([
            'documentable_id' => $project->id,
            'documentable_type' => Project::class,
        ]);

        $this->assertTrue($project->documents->contains($document));
    }

    #[Test]
public function it_has_project_expenses()
{
    $project = Project::factory()->create();
    $expense = Expense::factory()->forProject($project)->create();

    // Create a ProjectExpense record linking the project and expense
    $projectExpense = ProjectExpense::factory()->create([
        'project_id' => $project->id,
        'expense_id' => $expense->id,
    ]);

    // Refresh the project to load relationships
    $project->refresh();

    // Assert the projectExpenses relationship contains the projectExpense
    $this->assertTrue($project->projectExpenses->contains($projectExpense));
    
    // Verify the expense is linked through the projectExpense
    $this->assertEquals($expense->id, $projectExpense->expense_id);
}


    #[Test]
    public function it_can_have_project_clients()
    {
        $project = Project::factory()->create();
        $projectClient = ProjectClient::factory()->create(['project_id' => $project->id]);

        $this->assertTrue($project->projectClients->contains($projectClient));
    }

    #[Test]
    public function it_can_have_apartment_details()
    {
        $project = Project::factory()->create();
        $apartment = ApartmentDetail::factory()->create(['project_id' => $project->id]);

        $this->assertTrue($project->apartmentDetails->contains($apartment));
    }

    #[Test]
    public function it_can_have_subdivision_details()
    {
        $project = Project::factory()->create();
        $subdivision = SubDivisionDetail::factory()->create(['project_id' => $project->id]);

        $this->assertTrue($project->subdivisionDetails->contains($subdivision));
    }

    #[Test]
    public function it_can_have_invoices()
    {
        $project = Project::factory()->create();
        $invoice = Invoice::factory()->create([
            'payable_id' => $project->id,
            'payable_type' => Project::class,
        ]);

        $this->assertTrue($project->invoices->contains($invoice));
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        $project = Project::factory()->create();
        $project->delete();

        $this->assertSoftDeleted('projects', ['id' => $project->id]);
    }
}
