<?php

namespace Tests\Unit;

use App\Models\Client;
use App\Models\Document;
use App\Models\Payment;
use App\Models\Project;
use App\Models\ProjectClient;
use App\Models\ProjectClientPayment;
use App\Models\ApartmentDetail;
use App\Models\SubDivisionDetail;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class ProjectClientTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_has_fillable_attributes()
    {
        $projectClient = new ProjectClient();

        $this->assertEqualsCanonicalizing([
            'project_id',
            'client_id',
        ], $projectClient->getFillable());
    }

    #[Test]
    public function it_belongs_to_a_client()
    {
        $client = Client::factory()->create();
        $projectClient = ProjectClient::factory()->create([
            'client_id' => $client->id,
        ]);

        $this->assertInstanceOf(Client::class, $projectClient->client);
        $this->assertEquals($client->id, $projectClient->client->id);
    }

    #[Test]
    public function it_belongs_to_a_project()
    {
        $project = Project::factory()->create();
        $projectClient = ProjectClient::factory()->create([
            'project_id' => $project->id,
        ]);

        $this->assertInstanceOf(Project::class, $projectClient->project);
        $this->assertEquals($project->id, $projectClient->project->id);
    }

    #[Test]
    public function it_can_have_documents()
    {
        $projectClient = ProjectClient::factory()->create();
        $document = Document::factory()->create([
            'documentable_id' => $projectClient->id,
            'documentable_type' => ProjectClient::class,
        ]);

        $this->assertTrue($projectClient->documents->contains($document));
        $this->assertInstanceOf(Document::class, $projectClient->documents->first());
    }

    #[Test]
    public function it_can_have_project_client_payments()
    {
        $projectClient = ProjectClient::factory()->create();
        $payment = Payment::factory()->forProjectClient($projectClient)->create();
        $projectClientPayment = ProjectClientPayment::factory()->create([
            'project_client_id' => $projectClient->id,
            'payment_id' => $payment->id,
        ]);

        // Refresh the projectClient to load relationships
        $projectClient->refresh();

        // Assert the projectClientPayments relationship contains the projectClientPayment
        $this->assertTrue($projectClient->projectClientPayments->contains($projectClientPayment));
        $this->assertInstanceOf(ProjectClientPayment::class, $projectClient->projectClientPayments->first());
    }
    #[Test]
    public function it_can_have_an_apartment_detail()
    {
        $projectClient = ProjectClient::factory()->create();
        $apartmentDetail = ApartmentDetail::factory()->create([
            'project_client_id' => $projectClient->id,
        ]);

        $this->assertInstanceOf(ApartmentDetail::class, $projectClient->apartmentDetail);
        $this->assertEquals($apartmentDetail->id, $projectClient->apartmentDetail->id);
    }

    #[Test]
    public function it_can_have_a_subdivision_detail()
    {
        $projectClient = ProjectClient::factory()->create();
        $subdivisionDetail = SubDivisionDetail::factory()->create([
            'project_client_id' => $projectClient->id,
        ]);

        $this->assertInstanceOf(SubDivisionDetail::class, $projectClient->subdivisionDetail);
        $this->assertEquals($subdivisionDetail->id, $projectClient->subdivisionDetail->id);
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        $projectClient = ProjectClient::factory()->create();
        $projectClient->delete();

        $this->assertSoftDeleted('project_clients', ['id' => $projectClient->id]);
    }
}