<?php

namespace Tests\Unit;

use App\Models\Conveyancing;
use App\Models\Litigation;
use App\Models\Payment;
use App\Models\ProjectClient;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;


class PaymentTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_payment()
    {
        // Use Litigation to satisfy NOT NULL payable_id / payable_type
        $litigation = Litigation::factory()->create();
        $payment = Payment::factory()
            ->forLitigation($litigation)
            ->create([
                'payment_method' => 'cash',
                'status' => 'completed',
                'direction' => 'inflow',
            ]);

        $this->assertDatabaseHas('payments', [
            'id' => $payment->id,
            'payment_method' => 'cash',
            'status' => 'completed',
            'direction' => 'inflow',
        ]);
    }

    #[Test]
    public function it_can_attach_payment_to_conveyancing()
    {
        $conveyancing = Conveyancing::factory()->create();
        $payment = Payment::factory()->forConveyancing($conveyancing)->create();

        $this->assertEquals($conveyancing->id, $payment->payable_id);
        $this->assertEquals(Conveyancing::class, $payment->payable_type);
        $this->assertEquals(
            $conveyancing->buyer_id ?? $conveyancing->seller_id,
            $payment->client_id
        );
    }

    #[Test]
    public function it_can_attach_payment_to_litigation()
    {
        $litigation = Litigation::factory()->create();
        $payment = Payment::factory()->forLitigation($litigation)->create();

        $this->assertEquals($litigation->id, $payment->payable_id);
        $this->assertEquals(Litigation::class, $payment->payable_type);
        $this->assertEquals($litigation->client_id, $payment->client_id);
    }

    #[Test]
    public function it_can_attach_payment_to_project_client()
    {
        $projectClient = ProjectClient::factory()->create();
        $payment = Payment::factory()->forProjectClient($projectClient)->create();

        $this->assertEquals($projectClient->id, $payment->payable_id);
        $this->assertEquals(ProjectClient::class, $payment->payable_type);
        $this->assertEquals($projectClient->client_id, $payment->client_id);
    }

    #[Test]
    public function it_has_fillable_attributes()
    {
        $payment = new Payment();

        $this->assertEqualsCanonicalizing([
            'client_id',
            'payable_id',
            'payable_type',
            'amount_paid',
            'payment_method',
            'transaction_id',
            'direction',
            'status',
        ], $payment->getFillable());
    }
}
