<?php

namespace Tests\Unit;

use App\Models\Account;
use App\Models\Client;
use App\Models\Document;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\Invoice;
use App\Models\Litigation;
use App\Models\LitigationActivity;
use App\Models\LitigationAdvocate;
use App\Models\LitigationPayment;
use App\Models\LitigationExpense;
use App\Models\Payment;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class LitigationTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_generates_case_number_on_creation()
    {
        $client = Client::factory()->create();

        $litigation1 = Litigation::create([
            'client_id' => $client->id,
            'court_assigned_case_number' => 'CASE123',
            'type' => 'civil',
            'status' => 'open',
            'description' => 'Test litigation case',
            'case_category' => 'dispute',
            'station' => 'High Court',
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $this->assertEquals('AYC/LI/001', $litigation1->case_number);

        $litigation2 = Litigation::create([
            'client_id' => $client->id,
            'court_assigned_case_number' => 'CASE124',
            'type' => 'criminal',
            'status' => 'open',
            'description' => 'Another test case',
            'case_category' => 'appeal',
            'station' => 'Supreme Court',
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $this->assertEquals('AYC/LI/002', $litigation2->case_number);
    }

    #[Test]
    public function it_handles_soft_deletes()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::create([
            'client_id' => $client->id,
            'court_assigned_case_number' => 'CASE123',
            'type' => 'civil',
            'status' => 'open',
            'description' => 'Test litigation case',
            'case_category' => 'dispute',
            'station' => 'High Court',
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $litigation->delete();

        $this->assertSoftDeleted($litigation);

        $newLitigation = Litigation::create([
            'client_id' => $client->id,
            'court_assigned_case_number' => 'CASE124',
            'type' => 'civil',
            'status' => 'open',
            'description' => 'New test case',
            'case_category' => 'dispute',
            'station' => 'High Court',
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $this->assertEquals('AYC/LI/002', $newLitigation->case_number);
    }

    #[Test]
    public function it_belongs_to_a_client()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::factory()->create([
            'client_id' => $client->id,
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $this->assertInstanceOf(Client::class, $litigation->client);
        $this->assertEquals($client->id, $litigation->client->id);
    }

    #[Test]
    public function it_has_polymorphic_documents_relationship()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = Document::factory()->create([
            'name' => 'Court Filing',
            'file_path' => '/documents/court_filing.pdf',
            'documentable_type' => Litigation::class,
            'documentable_id' => $litigation->id,
            'serial_number' => 'AYC/LI/001/DOC/0001',
        ]);

        $this->assertNotNull($document, 'Document creation failed');
        $this->assertInstanceOf(Document::class, $litigation->documents->first());
        $this->assertEquals($document->id, $litigation->documents->first()->id);
        $this->assertEquals('Court Filing', $litigation->documents->first()->name);
        $this->assertEquals(Litigation::class, $document->documentable_type);
        $this->assertEquals($litigation->id, $document->documentable_id);
    }

    #[Test]
    public function it_has_many_payments()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $client = Client::factory()->create();

        LitigationPayment::factory()->create([
            'litigation_id' => $litigation->id,
            'client_id' => $client->id,
            'payment_id' => null,
            'amount_paid' => 1000.00,
            'payment_method' => 'mpesa',
            'transaction_id' => 'TXN-1234',
            'status' => 'completed',
            'type' => 'inflow',
        ]);
        LitigationPayment::factory()->create([
            'litigation_id' => $litigation->id,
            'client_id' => $client->id,
            'payment_id' => null,
            'amount_paid' => 500.00,
            'payment_method' => 'bank',
            'transaction_id' => 'TXN-5678',
            'status' => 'completed',
            'type' => 'inflow',
        ]);

        $this->assertCount(2, LitigationPayment::where('litigation_id', $litigation->id)->get(), 'LitigationPayment records not found in database');
        $litigation->load('litigationPayments');
        $this->assertCount(2, $litigation->litigationPayments, 'Litigation payments relationship empty');
        $this->assertInstanceOf(LitigationPayment::class, $litigation->litigationPayments->first());
    }

    #[Test]
    public function it_has_many_expenses()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $expenseItem = ExpenseItem::factory()->create();

        $expense1 = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => $expenseItem->id,
            'amount_paid' => 200.00,
            'balance' => 0.00,
            'payment_method' => 'cash',
            'status' => 'paid',
        ]);
        $expense2 = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => $expenseItem->id,
            'amount_paid' => 300.00,
            'balance' => 0.00,
            'payment_method' => 'bank',
            'status' => 'paid',
        ]);

        LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense1->id,
        ]);
        LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense2->id,
        ]);

        $this->assertCount(2, LitigationExpense::where('litigation_id', $litigation->id)->get(), 'LitigationExpense records not found in database');
        $litigation->load('litigationExpenses');
        $this->assertCount(2, $litigation->litigationExpenses, 'Litigation expenses relationship empty');
        $this->assertInstanceOf(LitigationExpense::class, $litigation->litigationExpenses->first());
    }

    #[Test]
    public function it_has_many_litigation_activities()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        LitigationActivity::factory()->create([
            'litigation_id' => $litigation->id,
        ]);
        LitigationActivity::factory()->create([
            'litigation_id' => $litigation->id,
        ]);

        $this->assertCount(2, $litigation->litigationActivities);
        $this->assertInstanceOf(LitigationActivity::class, $litigation->litigationActivities->first());
    }

    #[Test]
    public function it_has_many_invoices()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        Invoice::factory()->create([
            'payable_type' => Litigation::class,
            'payable_id' => $litigation->id,
            'client_id' => Client::factory()->create()->id,
            'issue_date' => now()->format('Y-m-d'),
            'due_date' => now()->addDays(30)->format('Y-m-d'),
        ]);
        Invoice::factory()->create([
            'payable_type' => Litigation::class,
            'payable_id' => $litigation->id,
            'client_id' => Client::factory()->create()->id,
            'issue_date' => now()->format('Y-m-d'),
            'due_date' => now()->addDays(30)->format('Y-m-d'),
        ]);

        $this->assertCount(2, $litigation->invoices);
        $this->assertInstanceOf(Invoice::class, $litigation->invoices->first());
    }

    #[Test]
    public function it_can_add_a_document()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = Document::factory()->create([
            'name' => 'Test Document',
            'file_path' => '/documents/test.pdf',
            'documentable_type' => Litigation::class,
            'documentable_id' => $litigation->id,
            'serial_number' => 'AYC/LI/001/DOC/0001',
        ]);

        $this->assertNotNull($document, 'Document creation failed');
        $this->assertInstanceOf(Document::class, $document);
        $this->assertEquals('Test Document', $document->name);
        $this->assertEquals('/documents/test.pdf', $document->file_path);
        $this->assertEquals(Litigation::class, $document->documentable_type);
        $this->assertEquals($litigation->id, $document->documentable_id);
        $this->assertTrue($litigation->documents->contains($document));
    }
}