<?php

namespace Tests\Unit;

use App\Models\Litigation;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\LitigationExpense;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class LitigationExpenseTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_litigation_expense()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $expenseItem = ExpenseItem::factory()->create();
        $expense = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => $expenseItem->id,
            'amount_paid' => 200.00,
            'balance' => 0.00,
            'payment_method' => 'cash',
            'status' => 'paid',
        ]);

        $litigationExpense = LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense->id,
        ]);

        $this->assertNotNull($litigationExpense);
        $this->assertEquals($litigation->id, $litigationExpense->litigation_id);
        $this->assertEquals($expense->id, $litigationExpense->expense_id);
    }

    #[Test]
    public function it_handles_soft_deletes()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $expenseItem = ExpenseItem::factory()->create();

        $expense = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => $expenseItem->id,
            'amount_paid' => 200.00,
            'balance' => 0.00,
            'payment_method' => 'cash',
            'status' => 'paid',
        ]);

        $litigationExpense = LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense->id,
        ]);

        $litigationExpense->delete();

        $this->assertSoftDeleted('litigation_expenses', [
            'id' => $litigationExpense->id,
        ]);
    }



    #[Test]
    public function it_belongs_to_a_litigation()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $expense = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => ExpenseItem::factory()->create()->id,
        ]);

        $litigationExpense = LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense->id,
        ]);

        $this->assertInstanceOf(Litigation::class, $litigationExpense->litigation);
        $this->assertEquals($litigation->id, $litigationExpense->litigation->id);
    }

    #[Test]
    public function it_belongs_to_an_expense()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $expense = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => ExpenseItem::factory()->create()->id,
        ]);

        $litigationExpense = LitigationExpense::factory()->create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense->id,
        ]);

        $this->assertInstanceOf(Expense::class, $litigationExpense->expense);
        $this->assertEquals($expense->id, $litigationExpense->expense->id);
    }

    #[Test]
    public function it_only_allows_fillable_fields_to_be_mass_assigned()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);
        $expense = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => ExpenseItem::factory()->create()->id,
        ]);

        $litigationExpense = LitigationExpense::create([
            'litigation_id' => $litigation->id,
            'expense_id' => $expense->id,
            'non_fillable_field' => 'Should not be assigned',
        ]);

        $this->assertNotNull($litigationExpense);
        $this->assertEquals($litigation->id, $litigationExpense->litigation_id);
        $this->assertEquals($expense->id, $litigationExpense->expense_id);
        $this->assertNull($litigationExpense->non_fillable_field);
    }
}