<?php

namespace Tests\Unit;

use App\Models\Litigation;
use App\Models\LitigationDocument;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class LitigationDocumentTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_litigation_document()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = LitigationDocument::factory()->forLitigation($litigation)->create([
            'name' => 'Court Filing',
            'file_path' => '/documents/court_filing.pdf',
            'serial_number' => 'AYC/LI/001/DOC/0001',
        ]);

        $this->assertNotNull($document);
        $this->assertInstanceOf(LitigationDocument::class, $document);
        $this->assertEquals('Court Filing', $document->name);
        $this->assertEquals('/documents/court_filing.pdf', $document->file_path);
        $this->assertEquals('AYC/LI/001/DOC/0001', $document->serial_number);
        $this->assertEquals($litigation->id, $document->documentable_id);
        $this->assertEquals(Litigation::class, $document->documentable_type);
    }

    #[Test]
    public function it_belongs_to_a_litigation()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = LitigationDocument::factory()->forLitigation($litigation)->create([
            'name' => 'Test Document',
            'file_path' => '/documents/test.pdf',
            'serial_number' => 'AYC/LI/001/DOC/0002',
        ]);

        $this->assertInstanceOf(Litigation::class, $document->litigation);
        $this->assertEquals($litigation->id, $document->litigation->id);
        $this->assertEquals($litigation->id, $document->documentable_id);
        $this->assertEquals(Litigation::class, $document->documentable_type);
    }

    #[Test]
    public function it_handles_soft_deletes_if_applicable()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = LitigationDocument::factory()->forLitigation($litigation)->create([
            'name' => 'Test Document',
            'file_path' => '/documents/test.pdf',
            'serial_number' => 'AYC/LI/001/DOC/0003',
        ]);

        $document->delete();

        if (method_exists($document, 'trashed')) {
            $this->assertSoftDeleted($document);
            $this->assertNotNull($document->deleted_at);
            $this->assertEquals(0, LitigationDocument::withoutTrashed()->count());
            $this->assertEquals(1, LitigationDocument::withTrashed()->count());
        } else {
            $this->assertNull($document->fresh());
            $this->assertEquals(0, LitigationDocument::count());
        }
    }

    #[Test]
    public function it_only_allows_fillable_fields_to_be_mass_assigned()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $document = LitigationDocument::create([
            'name' => 'Test Document',
            'file_path' => '/documents/test.pdf',
            'documentable_id' => $litigation->id,
            'documentable_type' => Litigation::class,
            'serial_number' => 'AYC/LI/001/DOC/0004',
            'non_fillable_field' => 'Should not be assigned',
        ]);

        $this->assertNotNull($document);
        $this->assertEquals('Test Document', $document->name);
        $this->assertEquals('/documents/test.pdf', $document->file_path);
        $this->assertEquals('AYC/LI/001/DOC/0004', $document->serial_number);
        $this->assertEquals($litigation->id, $document->documentable_id);
        $this->assertEquals(Litigation::class, $document->documentable_type);
        $this->assertNull($document->non_fillable_field);
    }
}