<?php

namespace Tests\Unit;

use App\Models\Litigation;
use App\Models\LitigationActivity;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class LitigationActivityTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_litigation_activity()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $activity = LitigationActivity::factory()->create([
            'litigation_id' => $litigation->id,
            'activity' => 'Court hearing scheduled',
            'activity_date' => '2025-08-20',
            'court_room' => 'Court 1',
            'outcome' => 'Pending',
        ]);

        $this->assertNotNull($activity);
        $this->assertEquals($litigation->id, $activity->litigation_id);
        $this->assertEquals('Court hearing scheduled', $activity->activity);
        $this->assertEquals('2025-08-20', $activity->activity_date);
        $this->assertEquals('Court 1', $activity->court_room);
        $this->assertEquals('Pending', $activity->outcome);
    }

    #[Test]
    public function it_handles_soft_deletes()
    {
        $activity = LitigationActivity::factory()->create([
            'activity_date' => now()->format('Y-m-d'),
        ]);

        $activity->delete();

        $this->assertSoftDeleted($activity);
        $this->assertNotNull($activity->deleted_at);
        $this->assertEquals(0, LitigationActivity::withoutTrashed()->count());
        $this->assertEquals(1, LitigationActivity::withTrashed()->count());
    }

    #[Test]
    public function it_belongs_to_a_litigation()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $activity = LitigationActivity::factory()->create([
            'litigation_id' => $litigation->id,
            'activity_date' => now()->format('Y-m-d'),
        ]);

        $this->assertInstanceOf(Litigation::class, $activity->litigation);
        $this->assertEquals($litigation->id, $activity->litigation->id);
    }

    #[Test]
    public function it_only_allows_fillable_fields_to_be_mass_assigned()
    {
        $litigation = Litigation::factory()->create([
            'filing_date' => now()->format('Y-m-d'),
        ]);

        $activity = LitigationActivity::create([
            'litigation_id' => $litigation->id,
            'activity' => 'Court hearing scheduled',
            'activity_date' => '2025-08-20',
            'court_room' => 'Court 1',
            'outcome' => 'Pending',
            'non_fillable_field' => 'Should not be assigned',
        ]);

        $this->assertNotNull($activity);
        $this->assertEquals($litigation->id, $activity->litigation_id);
        $this->assertEquals('Court hearing scheduled', $activity->activity);
        $this->assertEquals('2025-08-20', $activity->activity_date);
        $this->assertEquals('Court 1', $activity->court_room);
        $this->assertEquals('Pending', $activity->outcome);
        $this->assertNull($activity->non_fillable_field);
    }
}