<?php

namespace Tests\Unit;

use App\Models\Client;
use App\Models\Invoice;
use App\Models\Litigation;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use PHPUnit\Framework\Attributes\Test;

class InvoiceTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_generates_invoice_number_on_creation()
    {
        // Create a client
        $client = Client::factory()->create();

        // Create first invoice
        $invoice1 = Invoice::create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => 1,
            'status' => 'pending',
            'total_amount' => 1000.50,
        ]);

        // Check invoice number format (AYC/INV/00001)
        $this->assertEquals('AYC/INV/00001', $invoice1->invoice_number);

        // Create second invoice
        $invoice2 = Invoice::create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => 2,
            'status' => 'pending',
            'total_amount' => 2000.75,
        ]);

        // Check sequential invoice number (AYC/INV/00002)
        $this->assertEquals('AYC/INV/00002', $invoice2->invoice_number);
    }

    #[Test]
    public function it_handles_soft_deletes()
    {
        $client = Client::factory()->create();
        $invoice = Invoice::create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => 1,
            'status' => 'pending',
            'total_amount' => 1000.50,
        ]);

        // Soft delete the invoice
        $invoice->delete();

        // Check if invoice is soft deleted
        $this->assertSoftDeleted($invoice);

        // Verify invoice number generation respects soft-deleted invoices
        $newInvoice = Invoice::create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => 2,
            'status' => 'pending',
            'total_amount' => 1500.25,
        ]);

        // Should be AYC/INV/00002 since it counts soft-deleted invoice
        $this->assertEquals('AYC/INV/00002', $newInvoice->invoice_number);
    }

    #[Test]
    public function it_belongs_to_a_client()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::factory()->create();
        $invoice = Invoice::factory()->create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => $litigation->id,
        ]);

        $this->assertInstanceOf(Client::class, $invoice->client);
        $this->assertEquals($client->id, $invoice->client->id);
    }

    #[Test]
    public function it_has_polymorphic_payable_relationship()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::factory()->create();
        $invoice = Invoice::factory()->create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => $litigation->id,
        ]);

        $this->assertInstanceOf(Litigation::class, $invoice->payable);
        $this->assertEquals($litigation->id, $invoice->payable->id);
    }

    #[Test]
    public function it_casts_total_amount_to_two_decimal_places()
    {
        $client = Client::factory()->create();
        $invoice = Invoice::create([
            'client_id' => $client->id,
            'payable_type' => Litigation::class,
            'payable_id' => 1,
            'status' => 'pending',
            'total_amount' => 1000.5, // Input as 1000.5
        ]);

        // Should be cast to 1000.50
        $this->assertEquals('1000.50', $invoice->total_amount);
    }

    
}