<?php

namespace Tests\Unit;

use App\Models\Conveyancing;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\Litigation;
use App\Models\Project;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ExpenseTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_be_created_using_factory()
    {
        // Create an ExpenseItem and Conveyancing record
        $expenseItem = ExpenseItem::factory()->create();
        $conveyancing = Conveyancing::factory()->create();

        // Create an Expense instance
        $expense = Expense::factory()->forConveyancing($conveyancing)->create([
            'expense_item_id' => $expenseItem->id,
        ]);

        // Assert the record exists in the database
        $this->assertDatabaseHas('expenses', [
            'expense_item_id' => $expense->expense_item_id,
            'amount_paid' => $expense->amount_paid,
            'balance' => $expense->balance,
            'payment_method' => $expense->payment_method,
            'status' => $expense->status,
            'expensable_id' => $expense->expensable_id,
            'expensable_type' => $expense->expensable_type,
        ]);

        // Assert fields are not null
        $this->assertNotNull($expense->expense_item_id);
        $this->assertNotNull($expense->amount_paid);
        $this->assertNotNull($expense->balance);
        $this->assertNotNull($expense->payment_method);
        $this->assertNotNull($expense->status);
        $this->assertNotNull($expense->expensable_id);
        $this->assertNotNull($expense->expensable_type);
    }

    #[Test]
    public function it_belongs_to_expense_item()
    {
        $expenseItem = ExpenseItem::factory()->create();
        $conveyancing = Conveyancing::factory()->create();
        $expense = Expense::factory()->forConveyancing($conveyancing)->create([
            'expense_item_id' => $expenseItem->id,
        ]);

        $this->assertInstanceOf(ExpenseItem::class, $expense->expenseItem);
    }

    #[Test]
    public function it_belongs_to_expensable_conveyancing()
    {
        $conveyancing = Conveyancing::factory()->create();
        $expense = Expense::factory()->forConveyancing($conveyancing)->create();

        $this->assertInstanceOf(Conveyancing::class, $expense->expensable);
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        $conveyancing = Conveyancing::factory()->create();
        $expense = Expense::factory()->forConveyancing($conveyancing)->create();
        $expense->delete();

        $this->assertSoftDeleted($expense);
    }

    #[Test]
    public function it_belongs_to_expensable_litigation()
    {
        $litigation = Litigation::factory()->create();
        $expense = Expense::factory()->forLitigation($litigation)->create();

        $this->assertInstanceOf(Litigation::class, $expense->expensable);
    }

    #[Test]
    public function it_belongs_to_expensable_project()
    {
        $project = Project::factory()->create();
        $expense = Expense::factory()->forProject($project)->create();

        $this->assertInstanceOf(Project::class, $expense->expensable);
    }
}