<?php

namespace Tests\Unit;

use App\Models\Conveyancing;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\Litigation;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ExpenseItemTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_be_created_using_factory()
    {
        // Create an ExpenseItem instance
        $expenseItem = ExpenseItem::factory()->create();

        // Assert the record exists in the database
        $this->assertDatabaseHas('expense_items', [
            'expense_item' => $expenseItem->expense_item,
        ]);

        // Assert the field is not null
        $this->assertNotNull($expenseItem->expense_item);
    }

    #[Test]
    public function it_has_many_expenses()
    {
        // Create an ExpenseItem and Conveyancing record
        $expenseItem = ExpenseItem::factory()->create();
        $conveyancing = Conveyancing::factory()->create();

        // Create two Expense records associated with the ExpenseItem
        $expense1 = Expense::factory()->forConveyancing($conveyancing)->create([
            'expense_item_id' => $expenseItem->id,
        ]);
        $expense2 = Expense::factory()->forConveyancing($conveyancing)->create([
            'expense_item_id' => $expenseItem->id,
        ]);

        // Assert the expenses relationship
        $this->assertInstanceOf(\Illuminate\Database\Eloquent\Collection::class, $expenseItem->expenses);
        $this->assertCount(2, $expenseItem->expenses);
        $this->assertTrue($expenseItem->expenses->contains($expense1));
        $this->assertTrue($expenseItem->expenses->contains($expense2));
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        // Create an ExpenseItem instance
        $expenseItem = ExpenseItem::factory()->create();

        // Soft delete the ExpenseItem
        $expenseItem->delete();

        // Assert it is soft deleted
        $this->assertSoftDeleted($expenseItem);
    }

    #[Test]
    public function it_has_many_expenses_with_different_expensable_types()
    {
        // Create an ExpenseItem
        $expenseItem = ExpenseItem::factory()->create();

        // Create Conveyancing and Litigation records
        $conveyancing = Conveyancing::factory()->create();
        $litigation = Litigation::factory()->create();

        // Create Expense records for different expensable types
        $expense1 = Expense::factory()->forConveyancing($conveyancing)->create([
            'expense_item_id' => $expenseItem->id,
        ]);
        $expense2 = Expense::factory()->forLitigation($litigation)->create([
            'expense_item_id' => $expenseItem->id,
        ]);

        // Assert the expenses relationship includes both
        $this->assertInstanceOf(\Illuminate\Database\Eloquent\Collection::class, $expenseItem->expenses);
        $this->assertCount(2, $expenseItem->expenses);
        $this->assertTrue($expenseItem->expenses->contains($expense1));
        $this->assertTrue($expenseItem->expenses->contains($expense2));
    }
}