<?php

namespace Tests\Unit;

use App\Models\Conveyancing;
use App\Models\Document;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class DocumentTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_be_created_using_factory()
    {
        // Create a Conveyancing record first to ensure serial_number is set
        $conveyancing = Conveyancing::factory()->create();
        $document = Document::factory()->forConveyancing($conveyancing)->create();

        // Assert that the record exists in the database
        $this->assertDatabaseHas('documents', [
            'documentable_id' => $document->documentable_id,
            'documentable_type' => $document->documentable_type,
            'name' => $document->name,
            'file_path' => $document->file_path,
            'serial_number' => $document->serial_number,
        ]);

        // Assert the documentable relationship is loaded correctly
        $this->assertNotNull($document->documentable);
        $this->assertInstanceOf(Conveyancing::class, $document->documentable);
    }

    #[Test]
    public function it_belongs_to_documentable()
    {
        $conveyancing = Conveyancing::factory()->create();
        $document = Document::factory()->forConveyancing($conveyancing)->create();

        $this->assertInstanceOf(Conveyancing::class, $document->documentable);
    }

    #[Test]
    public function it_can_be_soft_deleted()
    {
        $conveyancing = Conveyancing::factory()->create();
        $document = Document::factory()->forConveyancing($conveyancing)->create();
        $document->delete();

        $this->assertSoftDeleted($document);
    }

    #[Test]
    public function it_generates_serial_number_correctly()
    {
        // Create a Conveyancing record with a specific ID
        $conveyancing = Conveyancing::factory()->create(['id' => 1]);

        // Ensure the Conveyancing serial_number is set
        $this->assertEquals('AYC/CV/001', $conveyancing->serial_number);

        $document1 = Document::factory()->forConveyancing($conveyancing)->create();

        // Assert the serial number follows the format AYC/CV/001/DOC/0001
        $this->assertEquals('AYC/CV/001/DOC/0001', $document1->serial_number);

        // Create another document for the same conveyancing to test sequence increment
        $document2 = Document::factory()->forConveyancing($conveyancing)->create();

        // Assert the serial number increments correctly
        $this->assertEquals('AYC/CV/001/DOC/0002', $document2->serial_number);
    }
}