<?php

namespace Tests\Unit;

use App\Models\{
    Advocate,
    Client,
    Conveyancing,
    ConveyancingAdvocate,
    ConveyancingDocument,
    ConveyancingExpense,
    ConveyancingPayment,
    Document,
    Invoice
};
use App\Models\Payment;
use Database\Factories\DocumentFactory;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ConveyancingTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_a_conveyancing()
    {
        $conveyancing = Conveyancing::factory()->create();

        $this->assertDatabaseHas('conveyancings', [
            'id' => $conveyancing->id,
        ]);
    }

    #[Test]
    public function it_can_soft_delete_a_conveyancing()
    {
        $conveyancing = Conveyancing::factory()->create();
        $conveyancing->delete();

        $this->assertSoftDeleted('conveyancings', [
            'id' => $conveyancing->id,
        ]);
    }

    #[Test]
    public function it_generates_a_serial_number_on_create()
    {
        $conveyancing = Conveyancing::factory()->create();

        $this->assertNotNull($conveyancing->serial_number);
        $this->assertMatchesRegularExpression('/^AYC\/CV\/\d{3}$/', $conveyancing->serial_number);
    }

    #[Test]
    public function it_can_update_a_conveyancing()
    {
        $conveyancing = Conveyancing::factory()->create();

        $conveyancing->update(['status' => 'completed']);

        $this->assertEquals('completed', $conveyancing->fresh()->status);
    }

    #[Test]
    public function it_has_conveyancing_documents()
    {
        $conveyancing = Conveyancing::factory()->create();

        // Use the DocumentFactory with the polymorphic state
        $doc = DocumentFactory::new()
            ->forConveyancing($conveyancing)
            ->create();

        $this->assertTrue(
            $conveyancing->conveyancingDocuments->contains($doc)
        );
    }


    #[Test]
    public function it_belongs_to_a_client()
    {
        $client = Client::factory()->create();
        $conveyancing = Conveyancing::factory()->create(['client_id' => $client->id]);

        $this->assertTrue($conveyancing->client->is($client));
    }

    #[Test]
    public function it_has_a_buyer_and_seller()
    {
        $buyer = Client::factory()->create();
        $seller = Client::factory()->create();
        $conveyancing = Conveyancing::factory()->create([
            'buyer_id' => $buyer->id,
            'seller_id' => $seller->id,
        ]);

        $this->assertTrue($conveyancing->buyer->is($buyer));
        $this->assertTrue($conveyancing->seller->is($seller));
    }

    #[Test]
    public function it_has_advocates_through_pivot()
    {
        $advocate = Advocate::factory()->create();
        $conveyancing = Conveyancing::factory()->create();

        $conveyancing->advocates()->attach($advocate->id, ['role' => 'buyer']);

        $this->assertTrue($conveyancing->advocates->contains($advocate));
        $this->assertEquals('buyer', $conveyancing->advocates->first()->pivot->role);
    }




    #[Test]
    public function it_has_conveyancing_payments()
    {
        $conveyancing = Conveyancing::factory()->create();

        // Use the polymorphic factory state
        $payment = Payment::factory()->forConveyancing($conveyancing)->create();

        $this->assertTrue($conveyancing->conveyancingPayments->contains($payment));
    }


    #[Test]
    public function it_has_conveyancing_expenses()
    {
        $conveyancing = Conveyancing::factory()->create();

        $expense = \App\Models\Expense::factory()
            ->forConveyancing($conveyancing)
            ->create();

        $this->assertTrue($conveyancing->expenses->contains($expense));
    }


    #[Test]
    public function it_has_invoices()
    {
        $conveyancing = Conveyancing::factory()->create();
        $invoice = Invoice::factory()->create([
            'payable_id' => $conveyancing->id,
            'payable_type' => Conveyancing::class,
        ]);

        $this->assertTrue($conveyancing->invoices->contains($invoice));
    }

    #[Test]
    public function it_has_documents()
    {
        $conveyancing = Conveyancing::factory()->create();
        $document = Document::factory()->create([
            'documentable_id' => $conveyancing->id,
            'documentable_type' => Conveyancing::class,
        ]);

        $this->assertTrue($conveyancing->documents->contains($document));
    }
}
