<?php

namespace Tests\Unit\Models;

use App\Models\Client;
use App\Models\User;
use App\Models\Payment;
use App\Models\Project;
use App\Models\ProjectClient;
use App\Models\Litigation;
use App\Models\LitigationPayment;
use App\Models\Conveyancing;
use App\Models\Quotation;
use App\Models\Document;
use App\Models\LitigationDocument;
use App\Models\ConveyancingDocument;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Storage;
use PHPUnit\Framework\Attributes\Test;
use Tests\TestCase;

class ClientTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_has_fillable_attributes()
    {
        $client = new Client();

        $this->assertEquals([
            'user_id',
            'phone',
            'national_id',
            'address',
            'passport_number',
            'nationality',
            'kra_pin',
            'identification_image',
        ], $client->getFillable());
    }

    #[Test]
    public function it_belongs_to_a_user()
    {
        $user = User::factory()->create();
        $client = Client::factory()->create(['user_id' => $user->id]);

        $this->assertInstanceOf(User::class, $client->user);
        $this->assertEquals($user->id, $client->user->id);
    }

    #[Test]
    public function it_has_many_payments()
    {
        $client = Client::factory()->create();
        $payment = Payment::factory()->create([
            'client_id' => $client->id,
            'payable_type' => ProjectClient::class,
            'payable_id' => ProjectClient::factory()->create()->id,
        ]);

        $this->assertTrue($client->payments->contains($payment));
    }

    #[Test]
    public function it_has_many_projects_through_project_clients()
    {
        $client = Client::factory()->create();
        $project = Project::factory()->create();
        ProjectClient::factory()->create([
            'client_id' => $client->id,
            'project_id' => $project->id,
        ]);

        $this->assertTrue($client->projects->contains($project));
    }

    #[Test]
    public function it_has_many_litigations()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::factory()->create(['client_id' => $client->id]);

        $this->assertTrue($client->litigations->contains($litigation));
    }

    #[Test]
    public function it_has_many_conveyancings()
    {
        $client = Client::factory()->create();
        $conveyancing = Conveyancing::factory()->create([
            'client_id' => $client->id,
            'status' => 'pending', // ✅ lowercase
        ]);

        $this->assertTrue($client->conveyancings->contains($conveyancing));
    }

    #[Test]
    public function it_has_many_quotations()
    {
        $client = Client::factory()->create();
        $quotation = Quotation::factory()->create(['client_id' => $client->id]);

        $this->assertTrue($client->quotations->contains($quotation));
    }

    #[Test]
    public function it_returns_identification_image_url_if_present()
    {
        Storage::fake('public');
        $client = Client::factory()->create(['identification_image' => 'id_docs/test.jpg']);

        $url = Storage::disk('public')->url('id_docs/test.jpg');

        $this->assertEquals($url, $client->identification_image_url);
    }

    #[Test]
    public function it_returns_null_identification_image_url_if_not_set()
    {
        $client = Client::factory()->create(['identification_image' => null]);

        $this->assertNull($client->identification_image_url);
    }

    #[Test]
    public function it_has_conveyancing_documents()
    {
        $client = Client::factory()->create();
        $conveyancing = Conveyancing::factory()->create([
            'client_id' => $client->id,
            'status' => 'pending',
        ]);

        $document = Document::factory()
            ->forConveyancing($conveyancing)
            ->create();

        $this->assertTrue($client->conveyancingDocuments->contains($document));
    }

    #[Test]
    public function it_has_litigation_documents()
    {
        $client = Client::factory()->create();
        $litigation = Litigation::factory()->create([
            'client_id' => $client->id,
        ]);

        $document = Document::factory()
            ->forLitigation($litigation)
            ->create();

        $this->assertTrue($client->litigationDocuments->contains($document));
    }

    #[Test]
    public function it_has_project_client_documents()
    {
        $client = Client::factory()->create();
        $projectClient = ProjectClient::factory()->create([
            'client_id' => $client->id,
        ]);

        $document = Document::factory()
            ->forProjectClient($projectClient)
            ->create();

        $this->assertTrue($client->projectClientDocuments->contains($document));
    }

    #[Test]
    public function it_merges_all_documents()
    {
        $client = Client::factory()->create();

        $conveyancing = Conveyancing::factory()->create([
            'client_id' => $client->id,
            'status' => 'pending',
        ]);
        $conveyancingDoc = Document::factory()
            ->forConveyancing($conveyancing)
            ->create();

        $litigation = Litigation::factory()->create(['client_id' => $client->id]);
        $litigationDoc = Document::factory()
            ->forLitigation($litigation)
            ->create();

        $projectClient = ProjectClient::factory()->create(['client_id' => $client->id]);
        $projectDoc = Document::factory()
            ->forProjectClient($projectClient)
            ->create();

        $allDocs = $client->allDocuments();

        $this->assertTrue($allDocs->contains('id', $conveyancingDoc->id));
        $this->assertTrue($allDocs->contains('id', $litigationDoc->id));
        $this->assertTrue($allDocs->contains('id', $projectDoc->id));

    }




}
