<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\ApartmentDetail;
use App\Models\Project;
use App\Models\ProjectClient;
use Illuminate\Foundation\Testing\RefreshDatabase;
use PHPUnit\Framework\Attributes\Test;

class ApartmentDetailTest extends TestCase
{
    use RefreshDatabase;

    #[Test]
    public function it_can_create_an_apartment_detail()
    {
        $client = ProjectClient::factory()->create();
        $apartment = ApartmentDetail::factory()->forClient($client)->create();

        $this->assertDatabaseHas('apartment_details', [
            'id' => $apartment->id,
            'flat_name' => $apartment->flat_name,
            'project_client_id' => $client->id,
            'project_id' => $client->project_id,
        ]);
    }

    #[Test]
    public function it_can_update_an_apartment_detail()
    {
        $client = ProjectClient::factory()->create();
        $apartment = ApartmentDetail::factory()->forClient($client)->create([
            'floor' => 1,
        ]);

        $apartment->update([
            'floor' => 2,
        ]);

        $this->assertDatabaseHas('apartment_details', [
            'id' => $apartment->id,
            'floor' => 2,
        ]);
    }

    #[Test]
    public function it_can_soft_delete_an_apartment_detail()
    {
        $client = ProjectClient::factory()->create();
        $apartment = ApartmentDetail::factory()->forClient($client)->create();

        $apartment->delete();

        $this->assertSoftDeleted('apartment_details', [
            'id' => $apartment->id,
        ]);
    }

    #[Test]
    public function an_apartment_detail_belongs_to_a_project()
    {
        $client = ProjectClient::factory()->create();
        $apartment = ApartmentDetail::factory()->forClient($client)->create();

        $this->assertInstanceOf(Project::class, $apartment->project);
        $this->assertEquals($client->project_id, $apartment->project->id);
    }

    #[Test]
    public function an_apartment_detail_belongs_to_a_project_client()
    {
        $client = ProjectClient::factory()->create();
        $apartment = ApartmentDetail::factory()->forClient($client)->create();

        $this->assertInstanceOf(ProjectClient::class, $apartment->projectClient);
        $this->assertEquals($client->id, $apartment->projectClient->id);
        $this->assertEquals($client->project_id, $apartment->project_id);
    }
}
