<?php


use Illuminate\Support\Facades\Route;
use App\Models\ProjectClient;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Payment;

//Let default path be the admin 
Route::redirect('/', '/admin'); 

Route::get('/dashboard', function () {
    return view('dashboard');
})->middleware(['auth', 'verified'])->name('dashboard');




Route::get('/projects/{record}/client/{projectClient}/pdf', function ($record, $projectClient) {
    $project = \App\Models\Project::findOrFail($record);
    $projectClient = ProjectClient::with([
        'client.user',
        'apartmentDetail',
        'subdivisionDetail',
        'documents',
        'projectClientPayments',
    ])->findOrFail($projectClient);

    if ($projectClient->project_id !== $project->id) {
        abort(403, 'Unauthorized action.');
    }

    \Illuminate\Support\Facades\Log::info('Export Client PDF Data:', [
        'project_id' => $project->id,
        'project_client_id' => $projectClient->id,
        'client_name' => $projectClient->client->user->name ?? 'N/A',
        'payments_count' => $projectClient->projectClientPayments->count(),
        'documents_count' => $projectClient->documents->count(),
    ]);

    $pdf = Pdf::loadView('pdf.project-client', [
        'project' => $project,
        'projectClient' => $projectClient,
    ]);

    $clientName = preg_replace('/[^A-Za-z0-9\-_]/', '_', $projectClient->client->user->name ?? 'client');
    $projectName = preg_replace('/[^A-Za-z0-9\-_]/', '_', $project->name);
    $fileName = "client-{$clientName}-{$projectName}-" . now()->format('Y-m-d') . ".pdf";

    return response()->streamDownload(
        fn () => print($pdf->output()),
        $fileName,
        [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => "attachment; filename={$fileName}",
        ]
    );
})->name('filament.resources.projects.client-pdf')->middleware('web');
Route::get('/admin/payments/get-entities', function(\Illuminate\Http\Request $request) {
    $controller = new \App\Filament\Resources\PaymentResource\Pages\CreatePointOfSale();
    $response = $controller->getEntities($request);
    return response()->json($response);
})->name('payments.get-entities');




Route::prefix('receipts')->name('receipts.')->group(function () {
   
    Route::get('/{payment}/download', function (Payment $payment) {
        // Load the payment with all required relationships
        $payment->load(['client.user', 'payable', 'account']);
       
        try {
            // Generate PDF
            $pdf = Pdf::loadView('reports.payment-receipt', ['payment' => $payment])
                ->setPaper('a4')
                ->setOptions([
                    'defaultFont' => 'sans-serif',
                    'isHtml5ParserEnabled' => true,
                    'isRemoteEnabled' => true
                ]);
           
            $fileName = 'payment_receipt_' . $payment->id . '_' . now()->format('Y-m-d_H-i-s') . '.pdf';
           
            // Return PDF as download
            return $pdf->download($fileName);
           
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to generate receipt',
                'message' => $e->getMessage()
            ], 500);
        }
    })->name('download');

    Route::get('/{payment}/view', function (Payment $payment) {
        // Load the payment with all required relationships
        $payment->load(['client.user', 'payable', 'account']);
       
        try {
            // Generate PDF
            $pdf = Pdf::loadView('reports.payment-receipt', ['payment' => $payment])
                ->setPaper('a4')
                ->setOptions([
                    'defaultFont' => 'sans-serif',
                    'isHtml5ParserEnabled' => true,
                    'isRemoteEnabled' => true
                ]);
           
            // Return PDF for inline viewing
            return $pdf->stream('payment_receipt_' . $payment->id . '.pdf');
           
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to generate receipt',
                'message' => $e->getMessage()
            ], 500);
        }
    })->name('view');
});

require __DIR__.'/auth.php';
