<?php

namespace Database\Seeders;

use App\Models\Client;
use App\Models\Litigation;
use App\Models\LitigationExpense;
use App\Models\LitigationPayment;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\Payment;
use Illuminate\Database\Seeder;
use Faker\Factory as Faker;

class LitigationSeeder extends Seeder
{
    public function run(): void
    {
        $faker = Faker::create();

        $clients = Client::inRandomOrder()->take(10)->pluck('id');

        $types = ['Civil', 'Family Law', 'Criminal', 'Employment'];
        $categories = ['Contract Dispute', 'Property', 'Inheritance', 'Personal Injury'];
        $statuses = ['Pending', 'Open', 'Closed', 'Appealed', 'Dismissed'];

        // Create default expense items if they don't exist
        $defaultExpenses = ['Court Filing Fee', 'Service Fee', 'Transport', 'Stationery'];
        $expenseItems = [];

        foreach ($defaultExpenses as $name) {
            $item = ExpenseItem::firstOrCreate(['expense_item' => $name]);
            $expenseItems[] = $item;
        }

        for ($i = 1; $i <= 10; $i++) {
            $clientId = $clients->random();

            // Step 1: Create the litigation
            $litigation = Litigation::create([
                'case_number' => null,
                'court_assigned_case_number' => 'HCCC/E' . str_pad($i, 3, '0', STR_PAD_LEFT) . '/' . now()->year,
                'type' => $types[array_rand($types)],
                'description' => fake()->company() . ' VS ' . fake()->company() . ' AND ' . fake()->company(),
                'case_category' => $categories[array_rand($categories)],
                'station' => fake()->city(),
                'status' => $statuses[array_rand($statuses)],
                'filing_date' => fake()->dateTimeBetween('-2 years', 'now')->format('Y-m-d'),
                'client_id' => $clientId,
            ]);

            // Step 2: Update case number
            $litigation->update([
                'case_number' => 'AYC/LI/' . str_pad($litigation->id, 3, '0', STR_PAD_LEFT),
            ]);

            // Step 3: Add at least 4 expenses
            foreach ($expenseItems as $item) {
                $expense = Expense::create([
                    'expense_item_id' => $item->id,
                    'amount_paid' => $faker->numberBetween(2000, 15000),
                    'balance' => $faker->numberBetween(0, 5000),
                    'payment_method' => $faker->randomElement(['mpesa', 'bank', 'cash']),
                    'status' => $faker->randomElement(['pending', 'completed', 'failed']),
                    'expensable_id' => $litigation->id,
                    'expensable_type' => Litigation::class,
                ]);

                LitigationExpense::create([
                    'litigation_id' => $litigation->id,
                    'expense_id' => $expense->id,
                ]);
            }

            // Step 4: Add 2 payments (main and sub)
            foreach (['filing_fee', 'legal_fee'] as $type) {
                $amount = $faker->numberBetween(10000, 80000);
                $paymentMethod = $faker->randomElement(['mpesa', 'bank', 'cash']);
                $status = $faker->randomElement(['pending', 'completed', 'failed']);

                // Create record in payments table
                $payment = Payment::create([
                    'client_id' => $clientId,
                    'payable_id' => $litigation->id,
                    'payable_type' => Litigation::class,
                    'amount_paid' => $amount,
                    'payment_method' => $paymentMethod,
                    'transaction_id' => strtoupper($faker->bothify('TXN######')),
                    'status' => $status,
                    'direction' => 'inbound',
                ]);

                // Create record in litigation_payments table
                LitigationPayment::create([
                    'litigation_id' => $litigation->id,
                    'client_id' => $clientId,
                    'amount_paid' => $amount,
                    'type' => $type,
                    'payment_id' => $payment->id,
                    'payment_method' => $paymentMethod,
                    'transaction_id' => $payment->transaction_id,
                    'status' => $status,
                ]);
            }
        }
    }
}