<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Conveyancing;
use App\Models\Client;
use App\Models\Payment;
use App\Models\ConveyancingPayment;
use App\Models\ConveyancingExpense;
use App\Models\Expense;
use App\Models\ExpenseItem;
use Faker\Factory as Faker;

class ConveyancingSeeder extends Seeder
{
    public function run(): void
    {
        $faker = Faker::create();

        $clientIds = Client::pluck('id')->toArray();

        if (count($clientIds) < 3) {
            $this->command->warn('At least 3 clients are required. Please seed more clients.');
            return;
        }

        // Create expense items first
        $expenseItemNames = ['Valuation Fee', 'Registration Fee', 'Stamp Duty', 'Facilitation Fee'];
        $expenseItems = [];

        foreach ($expenseItemNames as $name) {
            $item = ExpenseItem::create(['expense_item' => $name]);
            $expenseItems[] = $item;
        }

        for ($i = 0; $i < 10; $i++) {
            $clientSample = $faker->randomElements($clientIds, 3);

            $buyerLegalFees = $faker->numberBetween(50000, 100000);
            $sellerLegalFees = $faker->numberBetween(50000, 100000);
            $salePrice = $faker->randomFloat(2, 100000, 1000000);

            $conveyancing = Conveyancing::create([
                'client_id' => $clientSample[0],
                'buyer_id' => $clientSample[1],
                'seller_id' => $clientSample[2],
                'serial_number' => null,
                'property_address' => $faker->address,
                'sale_price' => $salePrice,
                'buyer_legal_fees' => $buyerLegalFees,
                'seller_legal_fees' => $sellerLegalFees,
                'contract_date' => $faker->dateTimeBetween('-3 months', 'now')->format('Y-m-d'),
                'completion_period' => $faker->dateTimeBetween('now', '+2 months')->format('Y-m-d'),
                'notes' => $faker->sentence(),
            ]);

            $conveyancing->update([
                'serial_number' => 'AYC/CV/' . str_pad($conveyancing->id, 3, '0', STR_PAD_LEFT),
            ]);

            $buyerUserId = optional($conveyancing->buyer)->user_id;
            $sellerUserId = optional($conveyancing->seller)->user_id;

            $buyerPayment = Payment::create([
                'client_id' => $conveyancing->client_id,
                'payable_id' => $conveyancing->id,
                'payable_type' => Conveyancing::class,
                'amount_paid' => $buyerLegalFees + $salePrice,
                'payment_method' => $faker->randomElement(['mpesa', 'bank', 'cash']),
                'transaction_id' => strtoupper($faker->bothify('TXN#######')),
                'direction' => 'inbound',
                'status' => 'completed',
            ]);

            $sellerPayment = Payment::create([
                'client_id' => $conveyancing->client_id,
                'payable_id' => $conveyancing->id,
                'payable_type' => Conveyancing::class,
                'amount_paid' => $sellerLegalFees,
                'payment_method' => $faker->randomElement(['mpesa', 'bank', 'cash']),
                'transaction_id' => strtoupper($faker->bothify('TXN#######')),
                'direction' => 'inbound',
                'status' => 'completed',
            ]);

            ConveyancingPayment::create([
                'conveyancing_id' => $conveyancing->id,
                'client_id' => $conveyancing->client_id,
                'amount_paid' => $buyerLegalFees + $salePrice,
                'paid_by' => $buyerUserId,
                'payment_id' => $buyerPayment->id,
                'payment_method' => $buyerPayment->payment_method,
                'transaction_id' => $buyerPayment->transaction_id,
                'status' => $buyerPayment->status,
                'type' => 'purchase_price',
            ]);

            ConveyancingPayment::create([
                'conveyancing_id' => $conveyancing->id,
                'client_id' => $conveyancing->client_id,
                'amount_paid' => $sellerLegalFees,
                'paid_by' => $sellerUserId,
                'payment_id' => $sellerPayment->id,
                'payment_method' => $sellerPayment->payment_method,
                'transaction_id' => $sellerPayment->transaction_id,
                'status' => $sellerPayment->status,
                'type' => 'legal_fees',
            ]);

            // Create expenses for this conveyancing using created expense items
            foreach ($expenseItems as $item) {
                $expense = Expense::create([
                    'expense_item_id' => $item->id,
                    'amount_paid' => $faker->numberBetween(5000, 30000),
                    'balance' => $faker->numberBetween(0, 5000),
                    'payment_method' => $faker->randomElement(['mpesa', 'bank', 'cash']),
                    'status' => $faker->randomElement(['pending', 'completed', 'failed']),
                    'expensable_id' => $conveyancing->id,
                    'expensable_type' => Conveyancing::class,
                ]);

                ConveyancingExpense::create([
                    'conveyancing_id' => $conveyancing->id,
                    'expense_id' => $expense->id,
                ]);
            }
        }
    }
}