<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use App\Traits\HasDocuments;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\SoftDeletes;
class Project extends Model
{
    //
    use HasFactory, HasDocuments,SoftDeletes;
    protected $fillable = ["name", "poc", "location", "description", "number_of_units", "type","serial_number"];

    // a Project has many Clients through ProjectClient (many-to-many)
    public function clients(): BelongsToMany
    {
        return $this->belongsToMany(Client::class, 'project_clients')
            ->withPivot(['id', /* other pivot fields, e.g., 'role' */])
            ->withTimestamps();
    }

    //a project has many project documents
   // Polymorphic relationship to documents
    public function documents(): MorphMany
    {
        return $this->morphMany(Document::class, 'documentable');
    }


    //a project has many project expenses
    public function projectExpenses(): HasMany
    {
        return $this->hasMany(ProjectExpense::class);
    }

    //direct relationship with expenses model
     public function expenses(): MorphMany
    {
        return $this->morphMany(Expense::class, 'expensable');
    }
    

    //relationship with project client documents
    public function projectClients(): HasMany
    {
        return $this->hasMany(ProjectClient::class);
    }

    //relationship with apartment details
    public function apartmentDetails(): HasMany
    {
        return $this->hasMany(ApartmentDetail::class);
    }

    //relationship with subdivision details
    public function subdivisionDetails(): HasMany
    {
        return $this->hasMany(SubDivisionDetail::class);
    }

    //relationship with invoive
    public function invoices()
    {
        return $this->morphMany(Invoice::class, 'payable');
    }

    // Auto-generate serial_number when created
protected static function booted()
{
    static::creating(function ($project) {
        if (empty($project->serial_number)) {
            DB::transaction(function () use ($project) {
                // Extract the max numeric suffix from existing serial_numbers like 'AYC/PJ/0002'
                $lastNumber = DB::table('projects')
                    ->where('serial_number', 'like', 'AYC/PJ/%')
                    ->selectRaw("MAX(CAST(SUBSTRING(serial_number, 8) AS UNSIGNED)) as max_number")
                    ->lockForUpdate()
                    ->value('max_number');

                $nextNumber = ($lastNumber ?? 0) + 1;

                $project->serial_number = 'AYC/PJ/' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
            });
        }
    });
}



}

