<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\HasDocuments;

class Litigation extends Model
{
    use HasFactory, SoftDeletes, HasDocuments;
    //fields to be added
    protected $fillable = [
        'case_number',
        'court_assigned_case_number',
        'description',
        'amount_charged',
        'station',
        'filing_date',
        'client_id',
        'litigation_case_type_id',
        'litigation_category_id',
    ];




    //relationship between litigation and client
    //a case belongs to client
    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }


    // Polymorphic relationship to documents
    public function documents(): MorphMany
    {
        return $this->morphMany(Document::class, 'documentable');
    }

    //relationship with litigation payments
    public function litigationPayments(): HasMany
    {
        return $this->hasMany(LitigationPayment::class);
    }

    //relationship with advocates
    public function litigationAdvocates(): HasMany
    {
        return $this->hasMany(LitigationAdvocate::class);
    }

    //relationship with expenses
    public function litigationExpenses(): HasMany
    {
        return $this->hasMany(LitigationExpense::class);
    }

     // ✅ Polymorphic relationship for expenses
    public function expenses(): MorphMany
    {
        return $this->morphMany(Expense::class, 'expensable');
    }
    //relationship with stages
    public function litigationActivities(): HasMany
    {
        return $this->hasMany(LitigationActivity::class);
    }


    //relationship between litigation and invoices
    public function invoices()
    {
        return $this->morphMany(Invoice::class, 'payable');
    }
    //relationship with litigation case type & litigation category
    public function litigationCaseType()
    {
        return $this->belongsTo(LitigationCaseType::class);
    }

    public function litigationCategory()
    {
        return $this->belongsTo(LitigationCategory::class);
    }



    //generate random case number
    protected static function booted()
    {
        static::created(function (Litigation $litigation) {
            if (empty($litigation->case_number)) {
                $litigation->case_number = 'AYC/LI/' . str_pad($litigation->id, 3, '0', STR_PAD_LEFT);
                $litigation->saveQuietly(); // avoids triggering events again
            }
        });
    }

    public function addDocument(array $attributes)
    {
        return $this->documents()->create(array_merge($attributes, [
            'documentable_type' => self::class
        ]));
    }


}
