<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    // Enables soft deleting for invoices (records aren't permanently deleted)
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     * These are the fields we can fill via create() or update() methods.
     */
    protected $fillable = [
        'client_id',         // References the client being invoiced
        'payable_type',      // Polymorphic relation type (Litigation, Conveyancing, or Project)
        'payable_id',        // ID of the related entity
        'invoice_number',    // Unique, auto-generated invoice reference
        'status',            // e.g., 'pending', 'paid', 'overdue'
        'total_amount',      // Total amount on the invoice
        'notes',             // Any additional comments or instructions
    ];

    /**
     * Attribute casting to ensure proper data types.
     */
    protected $casts = [
        'issue_date' => 'date',          // Casts issue_date to Carbon date
        'due_date' => 'date',            // Casts due_date to Carbon date
        'total_amount' => 'decimal:2',   // Ensures two decimal places
    ];

    /**
     * Boot method to hook into model events.
     * Automatically generates an invoice number before creation.
     */
    protected static function booted()
    {
        static::creating(function ($invoice) {
            if (empty($invoice->invoice_number)) {
                // Fetch the latest invoice (including soft-deleted ones)
                $latest = static::withTrashed()->latest('id')->first();

                // Determine the next number in sequence
                $nextNumber = $latest ? $latest->id + 1 : 1;

                // Generate invoice number like "INV-00001"
                $invoice->invoice_number = 'AYC/INV/' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
            }
        });
    }

    /**
     * Relationship: An invoice belongs to a single client.
     */
    public function client()
    {
        return $this->belongsTo(Client::class);
    }

    /**
     * Polymorphic relationship: An invoice may belong to a litigation,
     * conveyancing, or project record.
     */
    public function payable()
    {
        return $this->morphTo();
    }
}
