<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\DB;

class Document extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'documentable_id',
        'documentable_type',
        'name',
        'file_path',
        'serial_number'
    ];

    public function documentable(): MorphTo
    {
        return $this->morphTo();
    }

    protected static function booted()
    {
        static::creating(function (Document $doc) {
            // Skip if already provided
            if (!empty($doc->serial_number)) {
                return;
            }

            $parent = $doc->documentable;
            if (!$parent) {
                return;
            }

            // Map type-specific prefix generators
            $configMap = [
                \App\Models\Conveyancing::class => fn($parent) =>
                    $parent->serial_number ?? 'AYC/CV/' . str_pad($parent->id, 3, '0', STR_PAD_LEFT),

                \App\Models\Litigation::class => fn($parent) =>
                    $parent->case_number ?? 'AYC/LI/' . str_pad($parent->id, 3, '0', STR_PAD_LEFT),

                \App\Models\Project::class => fn($parent) =>
                    $parent->serial_number ?? 'AYC/PJ/' . str_pad($parent->id, 3, '0', STR_PAD_LEFT),

                \App\Models\ProjectClient::class => function ($parent) {
                    $project = $parent->project;
                    if (!$project) return null;
                    return ($project->serial_number ?? 'AYC/PJ/' . str_pad($project->id, 3, '0', STR_PAD_LEFT)) . '/CLNT';
                },
            ];

            if (!isset($configMap[$doc->documentable_type])) {
                return;
            }

            $prefix = $configMap[$doc->documentable_type]($parent);
            if (!$prefix) {
                return;
            }

            // Ensure parent has its own serial number if applicable
            if (property_exists($parent, 'serial_number') && empty($parent->serial_number)) {
                $parent->serial_number = $prefix;
                $parent->saveQuietly();
            }

            // Count existing documents with the same prefix to determine the next sequence
            $count = DB::table('documents')
                ->where('serial_number', 'like', $prefix . '/DOC/%')
                ->count();

            // Assign the next sequence
            $doc->serial_number = $prefix . '/DOC/' . str_pad($count + 1, 4, '0', STR_PAD_LEFT);
        });
    }
}