<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\HasDocuments;

class Conveyancing extends Model
{
    use HasFactory, SoftDeletes, HasDocuments;
    protected $table = 'conveyancings';

    protected $fillable = [
        'client_id',
        'property_address',
        'seller_id',
        'buyer_id',
        'buyer_legal_fees',
        'seller_legal_fees',
        'sale_price',
        'contract_date',
        'completion_period',
        'notes',
    ];

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    // ✅ If you want to access pivot fields via the pivot model:
    public function conveyancingAdvocates(): HasMany
    {
        return $this->hasMany(ConveyancingAdvocate::class);
    }

    //  Optional: for convenience, you can also define this:
    public function advocates(): BelongsToMany
    {
        return $this->belongsToMany(Advocate::class, 'conveyancing_advocate')
            ->withPivot('role')
            ->withTimestamps();
    }

    //relationship with conveyancing documents
    public function conveyancingDocuments(): HasMany
    {
        return $this->hasMany(Document::class, 'documentable_id')
            ->where('documentable_type', self::class);
    }


    //relationship with conveyancing payments
    public function conveyancingPayments(): HasMany
    {
        return $this->hasMany(ConveyancingPayment::class);
    }



    //relationship with conveyancing expenses
    public function conveyancingExpenses(): HasMany
    {
        return $this->hasMany(ConveyancingExpense::class);
    }

    public function expenses(): MorphMany
    {
        return $this->morphMany(Expense::class, 'expensable');
    }


    //relationship with invoice
    public function invoices()
    {
        return $this->morphMany(Invoice::class, 'payable');
    }


    //relationship with buyer
    public function buyer()
    {
        return $this->belongsTo(Client::class, 'buyer_id');
    }


    //relationship with seller
    public function seller()
    {
        return $this->belongsTo(Client::class, 'seller_id');
    }

    // Polymorphic relationship to documents
    public function documents(): MorphMany
    {
        return $this->morphMany(Document::class, 'documentable');
    }

    //generate serial number
    protected static function booted()
    {
        static::created(function (Conveyancing $conveyancing) {
            if (empty($conveyancing->serial_number)) {
                $conveyancing->serial_number = 'AYC/CV/' . str_pad($conveyancing->id, 3, '0', STR_PAD_LEFT);
                $conveyancing->saveQuietly(); // instead of updateQuietly
            }
        });
    }


}

