<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasManyThrough; // Correct import
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Storage;

class Client extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'phone',
        'national_id',
        'address',
        'passport_number',
        'nationality',
        'kra_pin',
        'identification_image'
    ];

    // Relationship between user and clients
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
    
    public function expenses():HasMany{
        return $this->hasMany(Expense::class);
    }

    // Relationship between clients and payments
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    // Existing projectPayments relationship (unchanged)
    public function projectPayments(): HasMany
    {
        return $this->payments()->where('payable_type', \App\Models\ProjectClient::class);
    }

    // New relationship for project client payments with correct type hint
    public function projectClientPayments(): HasManyThrough
    {
        return $this->hasManyThrough(
            ProjectClientPayment::class, // Target model
            ProjectClient::class,        // Intermediate model
            'client_id',                 // Foreign key on ProjectClient
            'project_client_id',         // Foreign key on ProjectClientPayment
            'id',                        // Local key on Client
            'id'                         // Local key on ProjectClient
        );
    }

    // Client ↔ Projects (many-to-many through ProjectClient)
    public function projects(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'project_clients', 'client_id', 'project_id')
            ->withTimestamps();
    }


    // Relationship between client and litigation
    public function litigations(): HasMany
    {
        return $this->hasMany(Litigation::class);
    }

    // Relationship with litigation payments
    public function litigationPayments(): HasMany
    {
        return $this->hasMany(LitigationPayment::class);
    }

    // Relationship with conveyancing
    public function conveyancings(): HasMany
    {
        return $this->hasMany(Conveyancing::class);
    }

    // Relationship with conveyancing payments
    public function conveyancingPayments(): HasManyThrough
    {
        return $this->hasManyThrough(
            \App\Models\ConveyancingPayment::class,
            \App\Models\Conveyancing::class,
            'client_id',
            'conveyancing_id',
            'id',
            'id'
        );
    }

    // Relationship with quotations
    public function quotations(): HasMany
    {
        return $this->hasMany(Quotation::class);
    }

    // Accessor for identification image URL
    public function getIdentificationImageUrlAttribute()
    {
        return $this->identification_image
            ? Storage::disk('public')->url($this->identification_image)
            : null;
    }


    //relationship with project clients
    public function projectClients(): HasMany
    {
        return $this->hasMany(ProjectClient::class, 'client_id');
    }

    // All conveyancing documents
    public function conveyancingDocuments(): HasManyThrough
    {
        return $this->hasManyThrough(
            Document::class,
            Conveyancing::class,
            'client_id',        // Foreign key on conveyancings
            'documentable_id',  // Foreign key on documents
            'id',               // Local key on clients
            'id'                // Local key on conveyancings
        )->where('documents.documentable_type', Conveyancing::class);
    }

    // All litigation documents
    public function litigationDocuments(): HasManyThrough
    {
        return $this->hasManyThrough(
            Document::class,
            Litigation::class,
            'client_id',        // Foreign key on litigations
            'documentable_id',  // Foreign key on documents
            'id',               // Local key on clients
            'id'                // Local key on litigations
        )->where('documents.documentable_type', Litigation::class);
    }

    // All project client documents
    public function projectClientDocuments(): HasManyThrough
    {
        return $this->hasManyThrough(
            Document::class,
            ProjectClient::class,
            'client_id',           // Foreign key on project_clients
            'documentable_id',     // Foreign key on documents
            'id',                  // Local key on clients
            'id'                   // Local key on project_clients
        )->where('documents.documentable_type', ProjectClient::class);
    }


    public function allDocuments()
    {
        return collect()
            ->merge($this->conveyancingDocuments()->get())
            ->merge($this->litigationDocuments()->get())
            ->merge($this->projectClientDocuments()->get())
            ->unique('id')
            ->values();
    }


}