<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreRoleRequest;
use App\Http\Requests\UpdateRoleRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\DB;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index():View
    {
        //view all the roles
        $roles = Role::orderBy("id", "DESC")->paginate(3);
        //return a view with the roles 
        return view('roles.index', compact("roles"));

    }

    /**
     * Show the form for creating a new resource.
     */
    public function create():View
    {
        //returns an array of all permissions
        $permissions = Permission::get();
        //return view with permissions that will be assigned to the roles
        return view("roles.create", compact("permissions"));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreRoleRequest $request):RedirectResponse
    {
        try {
            // Create a new role with the provided name from the request
            $role = Role::create([
                "name" => $request->name
            ]);

            // Retrieve the permissions based on the selected permission IDs from the request
            $permissions = Permission::whereIn("id", $request->permissions)
                ->get(['name']) // Only fetch the 'name' attribute
                ->toArray();

            // Assign the permissions to the newly created role
            $role->syncPermissions($permissions);

            // Redirect to the roles index page with a success message
            return redirect()->route('roles.index')->with("success", "New role added successfully");

        } catch (\Exception $e) {
            // Log the error message or handle it as needed
            \Log::error('Role creation failed: ' . $e->getMessage());

            // Redirect back with an error message
            return redirect()->back()->with('error', 'Failed to create the role. Please try again.');
        }
    }


    /**
     * Display the specified resource.
     */
    public function show(Role $role):View
    {
        // Retrieve the permissions associated with the given role
        // Joining 'permissions' with the pivot table 'role_has_permissions' 
        // on the permission ID, and filtering by the current role's ID
        $rolePermissions = Permission::join(
            "role_has_permissions",
            "permission_id",
            "=",
            "id"
        )
            ->where("role_id", $role->id)
            ->select("name") // Select only the permission names
            ->get();

        // Return the 'roles.show' view with the role and its associated permissions
        return view("roles.show", [
            'role' => $role,
            'rolePermissions' => $rolePermissions
        ]);
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Role $role):View
    {
        // Prevent deletion or editing of the 'Super Admin' role
        if ($role->name == "Super Admin") {
            // Abort the request and return a 403 Forbidden response with a custom message
            abort(403, "SUPER ADMIN ROLE CAN NOT BE DELETED!");
        }

        // Retrieve the IDs of all permissions currently assigned to the role
        $rolePermissions = DB::table("role_has_permissions")
            ->where("role_id", $role->id)
            ->pluck("permission_id") // Get only the 'permission_id' column values
            ->all(); // Convert the result into a plain array

        // Return the 'roles.edit' view with the following data:
        // - the role being edited
        // - a list of all available permissions (for displaying checkboxes or similar)
        // - the permission IDs currently assigned to the role (to pre-check those in the UI)
        return view("roles.edit", [
            "role" => $role,
            "permissions" => Permission::get(),
            "rolePermissions" => $rolePermissions
        ]);

    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateRoleRequest $request, Role $role):RedirectResponse
    {
        try {
            // Extract only the 'name' field from the validated request input
            $input = $request->only("name");

            // Update the role's name with the new input
            $role->update($input);

            // Retrieve the names of the permissions selected in the form
            // This fetches the permission names using their IDs from the request
            $permissions = Permission::whereIn("id", $request->permissions)
                ->get(["name"])   // Fetch only the 'name' column
                ->toArray();      // Convert the collection to a plain array

            // Sync the retrieved permissions with the role
            // This will attach new permissions and remove ones not present in the array
            $role->syncPermissions($permissions);

            // Redirect back to the previous page with a success flash message
            return redirect()->route('roles.index')->with("success", "Role was updated successfully.");

        } catch (\Exception $e) {
            // Log the error message for debugging
            \Log::error('Role update failed: ' . $e->getMessage());

            // Redirect back with an error message for the user
            return redirect()->back()->with("error", "Failed to update the role. Please try again.");
        }
    }



    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role):RedirectResponse
    {
        try {
            // Prevent deletion of the 'Super Admin' role as a safeguard
            if ($role->name == "Super Admin") {
                abort(403, "SUPER ADMIN ROLE CANNOT BE DELETED!");
            }

            // Prevent users from deleting a role they are currently assigned to
            if (auth()->user()->hasRole($role->name)) {
                abort(403, "CANNOT DELETE SELF-ASSIGNED ROLE");
            }

            // Proceed to delete the role from the database
            $role->delete();

            // Redirect to the roles index with a success message
            return redirect()->route("roles.index")->with("success", "Role was deleted successfully.");

        } catch (\Exception $e) {
            // Log the error for debugging purposes
            \Log::error('Failed to delete role: ' . $e->getMessage());

            // Redirect back with an error message
            return redirect()->back()->with("error", "An error occurred while deleting the role. Please try again.");
        }
    }

}
