<?php

namespace App\Filament\Resources\UserResource\Pages;

use App\Filament\Resources\UserResource;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Hash;
use Filament\Resources\Pages\Page;
use App\Models\User;

class ChangePassword extends Page
{
    use InteractsWithForms;

    protected static string $resource = UserResource::class;
    protected static string $view = 'filament.resources.user-resource.pages.change-password';

    public ?array $data = [];
    public User $user; // the user being edited

    public function mount(int $record): void
    {
        $this->user = User::findOrFail($record); // load user from URL
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('password')
                    ->label('New Password')
                    ->password()
                    ->required()
                    ->minLength(8)
                    ->same('password_confirmation'),

                TextInput::make('password_confirmation')
                    ->label('Confirm Password')
                    ->password()
                    ->required(),
            ])
            ->statePath('data');
    }

    public function submit(): void
    {
        $this->validate([
            'data.password' => ['required', 'min:8', 'same:data.password_confirmation'],
            'data.password_confirmation' => ['required'],
        ]);

        $this->user->password = Hash::make($this->data['password']);
        $this->user->save();

        Notification::make()
            ->title('Password changed successfully!')
            ->success()
            ->send();

        // Redirect back to the edit page of this user
        $this->redirect(
            static::getResource()::getUrl('edit', ['record' => $this->user->id])
        );
    }
}
