<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Facades\Filament;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Form;
use Filament\Forms\Components\Card;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\CheckboxList;
use Filament\Resources\Resource;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationGroup = 'Admin';
    protected static ?string $navigationIcon = 'heroicon-o-user';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Card::make([
                Section::make('User Details')
                    ->schema([
                        TextInput::make('name')->required()->maxLength(255),
                        TextInput::make('email')->email()->required()->maxLength(255)->unique(ignoreRecord: true),
                        //username field
                        TextInput::make('username')->nullable()->maxLength(255)->unique(ignoreRecord: true),
                        //role setup
                        // Select::make('role')
                        //     ->label('User Role')
                        //     ->options(
                        //         Role::query()
                        //             ->when(
                        //                 !Filament::auth()->user()->hasRole('Super Admin'),
                        //                 fn($query) => $query->where('name', '!=', 'Super Admin')
                        //             )
                        //             ->pluck('name', 'name')
                        //     )
                        //     ->required()
                        //     ->reactive(),
                        Select::make('role')
                            ->label('User Role')
                            ->options(
                                Role::query()
                                    ->when(
                                        !Filament::auth()->user()->hasRole('Super Admin'),
                                        fn($query) => $query->where('name', '!=', 'Super Admin')
                                    )
                                    ->pluck('name', 'name')
                            )
                            ->required()
                            ->reactive()
                            ->afterStateHydrated(function ($component, $record) {
                                // prefill role field from user's current role
                                if ($record) {
                                    $component->state($record->roles()->pluck('name')->first());
                                }
                            })
                            ->afterStateUpdated(function ($state, $record) {
                                // sync selected role to Spatie
                                if ($record && $state) {
                                    $record->syncRoles([$state]);
                                }
                            }),


                        TextInput::make('password')
                            ->password()
                            ->required(fn(string $context): bool => $context === 'create')
                            ->dehydrateStateUsing(fn($state) => filled($state) ? Hash::make($state) : null)
                            ->dehydrated(fn($state) => filled($state))
                            ->visible(fn(string $context): bool => $context === 'create')
                            ->label('Password'),

                    ])
                    ->columns(2),

                Section::make('Client Details')
                    ->schema([
                        TextInput::make('client_phone')->label('Phone')->required(),
                        TextInput::make('client_national_id')->label('National ID')->required(),
                        TextInput::make('client_address')
                            ->label('Address')
                            ->maxLength(255)
                            ->nullable(),

                        //KRA Pin    
                        TextInput::make('kra_pin')
                            ->label('Kra Pin')
                            ->maxLength(255)
                            ->nullable(),

                        //Passport Number
                        TextInput::make('passport_number')
                            ->label('Passport Number')
                            ->maxLength(255)
                            ->nullable(),
                        //Nationality
                        TextInput::make('nationality')
                            ->label('Nationality')
                            ->maxLength(255)
                            ->nullable(),


                        //identification image
                        FileUpload::make('identification_image')
                            ->disk('public')
                            ->directory('client-identifications')
                            ->image()
                            ->maxSize(2048)
                            ->acceptedFileTypes(['image/jpeg', 'image/png'])
                            ->nullable(),
                    ])
                    ->visible(fn($get) => $get('role') === 'client')
                    ->columns(2),

                Section::make('Advocate Details')
                    ->schema([
                        TextInput::make('advocate_license_number')->label('License Number')->placeholder('Eg. P105/1234/24'),
                        Select::make('advocate_specialization')
                            ->label('Specialization')
                            ->options([
                                'criminal' => 'Criminal Law',
                                'civil' => 'Civil Law',
                                'corporate' => 'Corporate Law',
                                'family' => 'Family Law',
                                'property' => 'Property Law',
                            ])
                            ->searchable()
                            ->required(),
                        TextInput::make('advocate_phone')->label('Phone')->required()->placeholder('Eg. 0712345678'), // Kenyan format,
                        TextInput::make('advocate_national_id')->label('National ID')->placeholder('Eg. 12345678'),
                        TextInput::make('advocate_address')->label('Address')->placeholder('Eg. Kileleshwa, Nairobi'),
                    ])
                    ->visible(fn($get) => $get('role') === 'advocate')
                    ->columns(2),

                Section::make('Permissions')->schema(function () {
                    $permissions = Permission::orderBy('name')->get();
                    $grouped = $permissions->groupBy(fn($perm) => explode('_', $perm->name)[0]);

                    $sections = [];
                    foreach ($grouped as $resource => $perms) {
                        $sections[] = Section::make('Permissions to ' . ucfirst($resource))
                            ->schema([
                                CheckboxList::make("permissions.{$resource}")
                                    ->label('')
                                    ->options($perms->pluck('name', 'id'))
                                    ->columns(2)
                                    ->afterStateHydrated(function ($component, $record) use ($perms) {
                                        if (!$record)
                                            return;
                                        $selected = $record->permissions()
                                            ->whereIn('id', $perms->pluck('id'))
                                            ->pluck('id')
                                            ->toArray();
                                        $component->state($selected);
                                    }),
                            ])
                            ->collapsible()
                            ->collapsed();
                    }

                    return $sections;
                })->collapsible()->collapsed(),

            ]),
        ])->statePath('data');
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->searchable(),
                //TextColumn::make('email')->searchable(),
                TextColumn::make('roles.name')->label('Roles')->badge()->sortable()->searchable(),
                TextColumn::make('created_at')->dateTime()->sortable(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->latest();
    }


    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
            'view' => Pages\ViewUser::route('/{record}'),
            'change-password' => Pages\ChangePassword::route('/{record}/change-password'),       
         ];
    }
}
