<?php

namespace App\Filament\Resources\QuotationResource\Pages;

use App\Filament\Resources\QuotationResource;
use App\Mail\QuotationMail;
use Filament\Actions\Action;
use Filament\Actions;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\ViewEntry;
use Filament\Infolists\Components\Section;
use Filament\Resources\Pages\ViewRecord;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Filament\Notifications\Notification;

class ViewQuotation extends ViewRecord
{
    protected static string $resource = QuotationResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make(),
            Actions\DeleteAction::make(),
            //pdf quotation
            Action::make('downloadQuotation')
                ->label('Download Quotation')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('success')
                ->action(function () {
                    $this->record->loadMissing(['client']); // load related models if needed

                    $pdf = Pdf::loadView('pdf.quotation', [
                        'quotation' => $this->record,
                    ]);

                    return response()->streamDownload(
                        fn () => print($pdf->stream()),
                        'quotation-' . $this->record->id . '.pdf'
                    );
                }),
                //send email to the person
               Action::make('sendMail')
                ->label('Send Quotation Email')
                ->icon('heroicon-m-paper-airplane')
                //->color('success')
                ->requiresConfirmation()
                ->action(function ($record) {
                    // Adjust email access depending on your relationships
                    $email = $record->client?->user?->email;

                    if (!$email) {
                        Notification::make()
                            ->title('No email address found for the client.')
                            ->danger()
                            ->send();

                        return;
                    }

                    Mail::to($email)->send(new QuotationMail($record));

                    Notification::make()
                       ->title('Quotation sent successfully to ' . $email)
                        ->success()
                        ->send();
                }),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            Section::make('Quotation Information')
                ->schema([
                    ViewEntry::make('quotation')
                        ->view('infolists.quotation-components.quotation-details')
                        ->viewData([
                            'quotation' => $this->record,
                        ]),
                ]),
        ]);
    }
}
