<?php

namespace App\Filament\Resources;

use App\Filament\Resources\QuotationResource\Pages;
use App\Filament\Resources\QuotationResource\RelationManagers;
use App\Models\Quotation;
use App\Models\Client;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class QuotationResource extends Resource
{
    protected static ?string $model = Quotation::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'Financial Management';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Quotation Details')->schema([
                    //quote number
                    TextInput::make('quote_number')
                        ->label('Quote Number')
                        ->readOnly()
                        ->disabled()
                        ->dehydrated(false)
                        ->visible(fn($record) => filled($record)),

                    // Client selector with inline creation
                    Select::make('client_id')
                        ->label('Client')
                        ->searchable()
                        ->preload()
                        ->options(
                            Client::with('user')->get()
                                ->mapWithKeys(
                                    fn($client) =>
                                    optional($client->user)->name
                                    ? [$client->id => $client->user->name]
                                    : []
                                )
                        )
                        ->required()
                        ->createOptionForm([
                            TextInput::make('user.name')->label('Name')->required(),
                            TextInput::make('phone')->required(),
                            TextInput::make('national_id')->label('National ID')->required(),
                            Textarea::make('address')->rows(2)->required(),
                        ])
                        ->createOptionUsing(function (array $data) {
                            // Create user and client record
                            $user = User::create([
                                'name' => $data['user']['name'],
                            ]);
                            $user->assignRole('client');

                            $client = Client::create([
                                'user_id' => $user->id,
                                'phone' => $data['phone'],
                                'national_id' => $data['national_id'],
                                'address' => $data['address'],
                            ]);

                            return $client->getKey();
                        }),

                    Select::make('type')
                        ->label('Type')
                        ->options([
                            'conveyancing' => 'Conveyancing',
                            'litigation' => 'Litigation',
                            'project' => 'Project',
                        ])
                        ->required(),

                    DatePicker::make('quote_date')
                        ->label('Quote Date')
                        ->default(now())
                        ->required(),

                    TextInput::make('amount')
                        ->label('Total Amount')
                        ->numeric()
                        ->prefix('KES')
                        ->required(),

                    Textarea::make('details')
                        ->label('Quotation Summary')
                        ->rows(4),
                // optional, for better dropdown styling   
                ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                //
                Tables\Columns\TextColumn::make('quote_number')->searchable(),
                Tables\Columns\TextColumn::make('client.user.name')->label('Client')->searchable(),
                Tables\Columns\TextColumn::make('type')->badge(),
                Tables\Columns\TextColumn::make('amount')->money('KES'),
                Tables\Columns\TextColumn::make('quote_date')->date(),
                Tables\Columns\TextColumn::make('status')->badge(),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),

            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListQuotations::route('/'),
            'create' => Pages\CreateQuotation::route('/create'),
            'view' => Pages\ViewQuotation::route('/{record}'),
            'edit' => Pages\EditQuotation::route('/{record}/edit'),
        ];
    }
}
