<?php

namespace App\Filament\Resources\ProjectResource\Pages;

use App\Filament\Resources\ProjectResource;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\ViewEntry;
use Filament\Actions\Action;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;

class ViewProject extends ViewRecord
{
    protected static string $resource = ProjectResource::class;

    public function infolist(Infolist $infolist): Infolist
    {
        Log::info('ViewProject Infolist Data:', [
            'project_id' => $this->record ? $this->record->id : null,
            'projectClients_count' => $this->record ? $this->record->projectClients->count() : 0,
            'projectExpenses_count' => $this->record ? $this->record->projectExpenses->count() : 0,
            'projectClientDocuments_count' => $this->record ? $this->record->projectClients->flatMap->documents->count() : 0,
            'projectClientPayments_count' => $this->record ? $this->record->projectClients->flatMap->projectClientPayments->count() : 0,
        ]);

        return $infolist->schema([
            Section::make('Project Details')->schema([
                ViewEntry::make('projectDetails')
                    ->view('infolists.project-components.project-details-table')
                    ->viewData([
                        'project' => $this->record,
                        'documents' => $this->record->documents,
                        'getIconPath' => fn($filePath) => $this->getProjectIconPath($filePath),
                    ]),
            ]),
            Section::make('General Project Expenses')->schema([
                ViewEntry::make('expenses')
                    ->view('infolists.project-components.project-expenses-table')
                    ->viewData([
                        'expenses' => $this->record->expenses, // ✅ use polymorphic relation
                        'getIconPath' => fn($filePath) => $this->getProjectIconPath($filePath),
                    ]),
            ]),

            Section::make('Project Clients')->schema([
                ViewEntry::make('projectClients')
                    ->view('infolists.project-components.project-clients-table')
                    ->viewData([
                        'projectRecord' => $this->record,
                        'projectClients' => $this->record->projectClients,
                    ]),
            ]),
            Section::make('Project Client Payments')->schema([
                ViewEntry::make('projectClientPayments')
                    ->view('infolists.project-components.project-client-payments-table')
                    ->viewData([
                        'projectRecord' => $this->record,
                        'projectClients' => $this->record->projectClients,
                    ]),
            ]),
        ]);
    }

    protected function getHeaderActions(): array
    {
        return [
            //total expenses
            Action::make('export_expenses_pdf')
                ->label('Export Expenses to PDF')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->action(function () {
                    $project = $this->record->load('projectExpenses');
                    $expenses = $project->projectExpenses;
                    $totalExpensesPaid = $expenses->sum('amount_paid');
                    $totalExpensesBalance = $expenses->sum('balance');

                    Log::info('Export Expenses PDF Data:', [
                        'project_id' => $project->id,
                        'expenses_count' => $expenses->count(),
                        'total_paid' => $totalExpensesPaid,
                        'total_balance' => $totalExpensesBalance,
                    ]);

                    $pdf = Pdf::loadView('pdf.project-expenses', [
                        'project' => $project,
                        'expenses' => $expenses,
                        'totalExpensesPaid' => $totalExpensesPaid,
                        'totalExpensesBalance' => $totalExpensesBalance,
                    ]);

                    return response()->streamDownload(
                        fn() => print ($pdf->output()),
                        "project-expenses-{$project->name}-" . now()->format('Y-m-d') . ".pdf"
                    );
                }),

            //financial summary    
            Action::make('export_project_details_pdf')
                ->label('Export Project Details to PDF')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->action(function () {
                    $project = $this->record->load([
                        'projectClients.client.user',
                        'projectClients.apartmentDetail',
                        'projectClients.subdivisionDetail',
                        'projectClients.documents',
                        'projectClients.projectClientPayments',
                        'documents',
                    ]);

                    $totalAmountCharged = $project->projectClients->sum(function ($client) {
                        return $client->apartmentDetail?->amount_charged ?? $client->subdivisionDetail?->amount_charged ?? 0;
                    });


                    Log::info('Export Project Details PDF Data:', [
                        'project_id' => $project->id,
                        'clients_count' => $project->projectClients->count(),
                        'documents_count' => $project->documents->count(),
                        'client_documents_count' => $project->projectClients->flatMap->documents->count(),
                        'client_payments_count' => $project->projectClients->flatMap->projectClientPayments->count(),
                    ]);

                    $pdf = Pdf::loadView('pdf.project', [
                        'project' => $project,
                        'totalAmountCharged' => $totalAmountCharged,

                    ]);

                    return response()->streamDownload(
                        fn() => print ($pdf->output()),
                        "project-details-{$project->name}-" . now()->format('Y-m-d') . ".pdf"
                    );
                }),

            //EXport client payment summary    
            Action::make('export_client_pdf')
                ->label('Export Client PDF')
                ->hidden() // Hide from header actions since it's triggered via table link
                ->url(fn($livewire, $arguments) => route('filament.resources.projects.client-pdf', [
                    'record' => $livewire->record->id,
                    'projectClient' => $arguments['projectClientId'],
                ]))
                ->openUrlInNewTab()
                ->icon('heroicon-o-arrow-down-tray'),


            Action::make('edit')
                ->label('Edit')
                ->icon('heroicon-o-pencil')
                ->url(fn() => ProjectResource::getUrl('edit', ['record' => $this->record])),
        ];
    }

    public function getProjectIconPath(string $filePath): string
    {
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
        return match ($extension) {
            'pdf' => asset('images/pdf-icon.png'),
            'doc', 'docx' => asset('images/word-icon.png'),
            'jpg', 'jpeg', 'png', 'gif' => asset('images/image-icon.png'),
            default => asset('images/file-icon.png'),
        };
    }
}