<?php

namespace App\Filament\Resources\ProjectResource\Pages;

use App\Filament\Resources\ProjectResource;
use App\Models\Project;
use App\Models\ProjectClient;
use Filament\Actions;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Models\Client;
use App\Models\User;

class EditProject extends EditRecord
{
    protected static string $resource = ProjectResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\ViewAction::make(),
            Actions\DeleteAction::make(),
        ];
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $project = $this->record;

        $data['project_id'] = $project->id;

        $data['documents'] = $project->documents()->where('documentable_type', Project::class)->get()->map(function ($doc) {
            return [
                'id' => $doc->id,
                'name' => $doc->name,
                'file_path' => [$doc->file_path],
            ];
        })->toArray();

        $data['projectClients'] = $project->projectClients()->with([
            'client.user',
            'apartmentDetail',
            'subdivisionDetail',
            'documents' => fn ($query) => $query->where('documentable_type', ProjectClient::class),
        ])->get()->map(function ($projectClient) {
            $clientData = [
                'id' => $projectClient->id,
                'client_id' => $projectClient->client_id,
                'passport_number' => $projectClient->client->passport_number,
                'national_id' => $projectClient->client->national_id,
                'address' => $projectClient->client->address,
                'phone' => $projectClient->client->phone,
                'nationality' => $projectClient->client->nationality,
                'kra_pin' => $projectClient->client->kra_pin,
                'identification_image' => $projectClient->client->identification_image ? [$projectClient->client->identification_image] : null,
                'email' => $projectClient->client->user->email ?? null,
                'documents' => $projectClient->documents->map(function ($doc) {
                    return [
                        'id' => $doc->id,
                        'name' => $doc->name,
                        'file_path' => [$doc->file_path],
                    ];
                })->toArray(),
            ];

            if ($projectClient->apartmentDetail) {
                $clientData['apartmentDetail'] = [
                    'id' => $projectClient->apartmentDetail->id,
                    'floor' => $projectClient->apartmentDetail->floor,
                    'flat_name' => $projectClient->apartmentDetail->flat_name,
                    'unit_type' => $projectClient->apartmentDetail->unit_type,
                    'owner2' => $projectClient->apartmentDetail->owner2,
                    'amount_charged' => $projectClient->apartmentDetail->amount_charged,
                    'project_id' => $projectClient->apartmentDetail->project_id,
                ];
            } elseif ($projectClient->subdivisionDetail) {
                $clientData['subdivisionDetail'] = [
                    'id' => $projectClient->subdivisionDetail->id,
                    'plot_number' => $projectClient->subdivisionDetail->plot_number,
                    'size' => $projectClient->subdivisionDetail->size,
                    'owner2' => $projectClient->subdivisionDetail->owner2,
                    'amount_charged' => $projectClient->subdivisionDetail->amount_charged,
                    'project_id' => $projectClient->subdivisionDetail->project_id,
                ];
            }

            return $clientData;
        })->toArray();

        Log::info('Form data before fill:', ['data' => $data]);

        return $data;
    }

    protected function handleRecordUpdate($record, array $data): \Illuminate\Database\Eloquent\Model
    {
        return DB::transaction(function () use ($record, $data) {
            try {
                Log::info('Form data for update:', ['data' => $data]);

                // Handle project documents
                if (isset($data['documents'])) {
                    $this->handleProjectDocuments($record, $data['documents']);
                }

                // Handle project clients
                $existingClientIds = $record->projectClients()->pluck('id')->toArray();
                $newClientIds = [];

                if (!empty($data['projectClients'])) {
                    foreach ($data['projectClients'] as $index => $clientData) {
                        Log::info('Processing client data:', ['index' => $index, 'clientData' => $clientData]);

                        $identificationImage = null;
                        if (!empty($clientData['identification_image'])) {
                            Log::info('Processing identification_image for client:', ['index' => $index, 'identification_image' => $clientData['identification_image']]);
                            if (is_array($clientData['identification_image'])) {
                                $firstFile = reset($clientData['identification_image']);
                                if ($firstFile instanceof TemporaryUploadedFile) {
                                    if (!$firstFile->exists()) {
                                        throw new \Exception('Temporary file for identification image does not exist.');
                                    }
                                    $identificationImage = $firstFile->store('client-identifications', 'public');
                                    Log::info('Identification image stored:', ['path' => $identificationImage]);
                                } else {
                                    $identificationImage = $firstFile;
                                }
                            } else {
                                $identificationImage = $clientData['identification_image'];
                            }
                        }

                        $client = null;
                        if (isset($clientData['client_id']) && $clientData['client_id']) {
                            $client = Client::find($clientData['client_id']);
                            if ($client) {
                                $client->update([
                                    'phone' => $clientData['phone'] ?? $client->phone,
                                    'national_id' => $clientData['national_id'] ?? $client->national_id,
                                    'address' => $clientData['address'] ?? $client->address,
                                    'nationality' => $clientData['nationality'] ?? $client->nationality,
                                    'passport_number' => $clientData['passport_number'] ?? $client->passport_number,
                                    'kra_pin' => $clientData['kra_pin'] ?? $client->kra_pin,
                                    'identification_image' => $identificationImage ?? $client->identification_image,
                                ]);

                                $emailToUpdate = $clientData['email'] ?? null;
                                if ($emailToUpdate && $client->user) {
                                    if (User::where('email', $emailToUpdate)->where('id', '!=', $client->user->id)->exists()) {
                                        Log::warning('Email update failed due to duplicate:', [
                                            'client_id' => $client->id,
                                            'email' => $emailToUpdate,
                                        ]);
                                        throw new \Exception('The email "' . $emailToUpdate . '" is already in use by another user.');
                                    }
                                    $client->user->update([
                                        'email' => $emailToUpdate,
                                    ]);
                                    Log::info('Email updated for user:', [
                                        'user_id' => $client->user->id,
                                        'new_email' => $emailToUpdate,
                                    ]);
                                }
                            } else {
                                Log::warning('Client not found for ID:', ['client_id' => $clientData['client_id']]);
                                throw new \Exception('Client not found for ID: ' . $clientData['client_id']);
                            }
                        } else {
                            $emailToCreate = $clientData['user']['email'] ?? null;
                            if (User::where('email', $emailToCreate)->exists()) {
                                Log::warning('Client creation failed due to duplicate email:', [
                                    'email' => $emailToCreate,
                                ]);
                                throw new \Exception('The email "' . $emailToCreate . '" is already in use.');
                            }
                            $user = User::create([
                                'name' => $clientData['user']['name'] ?? 'Unknown',
                                'email' => $emailToCreate ?? 'unknown_' . time() . '@example.com',
                            ]);
                            $user->assignRole('client');

                            $client = Client::create([
                                'user_id' => $user->id,
                                'phone' => $clientData['phone'] ?? null,
                                'national_id' => $clientData['national_id'] ?? null,
                                'address' => $clientData['address'] ?? null,
                                'nationality' => $clientData['nationality'] ?? null,
                                'passport_number' => $clientData['passport_number'] ?? null,
                                'kra_pin' => $clientData['kra_pin'] ?? null,
                                'identification_image' => $identificationImage,
                            ]);
                            Log::info('Client created:', ['client_id' => $client->id]);
                            $clientData['client_id'] = $client->id;
                        }

                        $clientRecordData = [
                            'client_id' => $client->id,
                        ];

                        if (isset($clientData['id']) && $record->projectClients()->where('id', $clientData['id'])->exists()) {
                            $projectClient = $record->projectClients()->where('id', $clientData['id'])->first();
                            $projectClient->update($clientRecordData);
                            $newClientIds[] = $projectClient->id;
                        } else {
                            $projectClient = $record->projectClients()->create($clientRecordData);
                            $newClientIds[] = $projectClient->id;
                        }

                        if ($record->type === 'apartment' && isset($clientData['apartmentDetail'])) {
                            $apartmentData = [
                                'project_id' => $record->id,
                                'project_client_id' => $projectClient->id,
                                'floor' => $clientData['apartmentDetail']['floor'],
                                'flat_name' => $clientData['apartmentDetail']['flat_name'],
                                'unit_type' => $clientData['apartmentDetail']['unit_type'],
                                'owner2' => $clientData['apartmentDetail']['owner2'] ?? null,
                                'amount_charged' => $clientData['apartmentDetail']['amount_charged'],
                            ];

                            if (isset($clientData['apartmentDetail']['id']) && $projectClient->apartmentDetail()->where('id', $clientData['apartmentDetail']['id'])->exists()) {
                                $projectClient->apartmentDetail()->where('id', $clientData['apartmentDetail']['id'])->update($apartmentData);
                            } else {
                                $projectClient->apartmentDetail()->create($apartmentData);
                            }
                        }

                        if ($record->type === 'sub_division' && isset($clientData['subdivisionDetail'])) {
                            $subdivisionData = [
                                'project_id' => $record->id,
                                'project_client_id' => $projectClient->id,
                                'plot_number' => $clientData['subdivisionDetail']['plot_number'],
                                'size' => $clientData['subdivisionDetail']['size'],
                                'owner2' => $clientData['subdivisionDetail']['owner2'] ?? null,
                                'amount_charged' => $clientData['subdivisionDetail']['amount_charged'],
                            ];

                            if (isset($clientData['subdivisionDetail']['id']) && $projectClient->subdivisionDetail()->where('id', $clientData['subdivisionDetail']['id'])->exists()) {
                                $projectClient->subdivisionDetail()->where('id', $clientData['subdivisionDetail']['id'])->update($subdivisionData);
                            } else {
                                $projectClient->subdivisionDetail()->create($subdivisionData);
                            }
                        }

                        if (!empty($clientData['documents'])) {
                            $existingDocIds = $projectClient->documents()->where('documentable_type', ProjectClient::class)->pluck('id')->toArray();
                            $newDocIds = [];

                            foreach ($clientData['documents'] as $docIndex => $document) {
                                $filePath = null;
                                if (is_array($document['file_path'])) {
                                    $firstFile = reset($document['file_path']);
                                    if ($firstFile instanceof TemporaryUploadedFile) {
                                        if (!$firstFile->exists()) {
                                            throw new \Exception('Temporary file for client document does not exist.');
                                        }
                                        $filePath = $firstFile->store('project-client-documents', 'public');
                                        Log::info('Client document stored:', ['path' => $filePath]);
                                    } else {
                                        $filePath = $firstFile;
                                    }
                                } else {
                                    $filePath = $document['file_path'];
                                }

                                if ($filePath === null) {
                                    continue;
                                }

                                $docData = [
                                    'name' => $document['name'],
                                    'file_path' => $filePath,
                                    'documentable_id' => $projectClient->id,
                                    'documentable_type' => ProjectClient::class,
                                ];

                                if (isset($document['id']) && $projectClient->documents()->where('id', $document['id'])->exists()) {
                                    $projectClient->documents()->where('id', $document['id'])->update($docData);
                                    $newDocIds[] = $document['id'];
                                } else {
                                    $newDoc = $projectClient->documents()->create($docData);
                                    $newDocIds[] = $newDoc->id;
                                }
                            }

                            $docsToDelete = array_diff($existingDocIds, $newDocIds);
                            $projectClient->documents()->whereIn('id', $docsToDelete)->delete();
                        }
                    }
                }

                $clientsToDelete = array_diff($existingClientIds, $newClientIds);
                if (!empty($clientsToDelete)) {
                    $record->projectClients()->whereIn('id', $clientsToDelete)->delete();
                }

                return $record;
            } catch (\Exception $e) {
                Log::error('Failed to update project clients: ' . $e->getMessage());
                Notification::make()
                    ->title('Error')
                    ->body('Failed to update project clients: ' . $e->getMessage())
                    ->danger()
                    ->send();
                throw $e;
            }
        });
    }

    protected function handleProjectDocuments($record, array $documentsData): void
    {
        $existingDocIds = $record->documents()->where('documentable_type', Project::class)->pluck('id')->toArray();
        $newDocIds = [];

        foreach ($documentsData as $document) {
            $filePath = null;
            if (is_array($document['file_path'])) {
                $firstFile = reset($document['file_path']);
                if ($firstFile instanceof TemporaryUploadedFile) {
                    if (!$firstFile->exists()) {
                        throw new \Exception('Temporary file for project document does not exist.');
                    }
                    $filePath = $firstFile->store('project-documents', 'public');
                    Log::info('Project document stored:', ['path' => $filePath]);
                } else {
                    $filePath = $firstFile;
                }
            } else {
                $filePath = $document['file_path'];
            }

            if ($filePath === null) {
                continue;
            }

            $docData = [
                'name' => $document['name'],
                'file_path' => $filePath,
                'documentable_id' => $record->id,
                'documentable_type' => Project::class,
            ];

            if (isset($document['id']) && $record->documents()->where('id', $document['id'])->exists()) {
                $record->documents()->where('id', $document['id'])->update($docData);
                $newDocIds[] = $document['id'];
            } else {
                $newDoc = $record->documents()->create($docData);
                $newDocIds[] = $newDoc->id;
            }
        }

        $docsToDelete = array_diff($existingDocIds, $newDocIds);
        if (!empty($docsToDelete)) {
            $record->documents()->whereIn('id', $docsToDelete)->delete();
        }
    }

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title('Project updated')
            ->body('The project and its clients have been updated successfully.');
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('view', ['record' => $this->getRecord()]);
    }
}