<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProjectResource\Pages;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Models\Client;
use App\Models\Project;
use App\Models\ProjectClient;
use App\Models\User;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Wizard;
use Filament\Forms\Components\Wizard\Step;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Filament\Notifications\Notification;

class ProjectResource extends Resource
{
    protected static ?string $model = Project::class;

    protected static ?int $navigationSort = 1;

    protected static ?string $navigationIcon = 'heroicon-o-briefcase';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Wizard::make([
                Step::make('Project Info & Documents')
                    ->schema([
                        Section::make('Project Info')
                            ->columnSpanFull()
                            ->schema([
                                Grid::make(2)->schema([
                                    TextInput::make('serial_number')
                                        ->label('Project Number')
                                        ->hidden()
                                        ->dehydrated() // keeps sending value to backend
                                        ->default(fn($record) => $record?->serial_number),

                                    TextInput::make('name')
                                        ->required()
                                        ->maxLength(255),
                                    TextInput::make('location')
                                        ->required()
                                        ->maxLength(255),
                                    TextInput::make('poc')
                                        ->label('Person of Contact')
                                        ->required()
                                        ->maxLength(255),
                                    Textarea::make('description')
                                        ->rows(3)
                                        ->columnSpanFull()
                                        ->nullable(),
                                    Select::make('type')
                                        ->options([
                                            'apartment' => 'Apartment',
                                            'sub_division' => 'Subdivision',
                                        ])
                                        ->required()
                                        ->reactive(),
                                    TextInput::make('number_of_units')
                                        ->label('Number of Units | Plots')
                                        ->numeric()
                                        ->minValue(0)
                                        ->nullable(),
                                ]),
                            ]),

                        Section::make('General Documents')
                            ->columnSpanFull()
                            ->schema([
                                Repeater::make('documents')
                                    ->relationship('documents')
                                    ->label('General Project Documents')
                                    ->schema([
                                        Grid::make(2)->schema([
                                            TextInput::make('name')
                                                ->required()
                                                ->maxLength(255),
                                            FileUpload::make('file_path')
                                                ->disk('public')
                                                ->directory('project-documents')
                                                ->required()
                                                ->acceptedFileTypes(['application/pdf', 'image/*'])
                                                ->maxSize(5120)
                                                ->multiple(false)
                                                ->afterStateUpdated(function ($state) {
                                                    Log::info('FileUpload state updated:', ['file_path' => $state]);
                                                }),
                                        ]),
                                    ])
                                    ->collapsible()
                                    ->defaultItems(1)
                                    ->minItems(1)
                                    ->required(),
                            ]),
                        Hidden::make('project_id')
                            ->default(fn(callable $get) => $get('project_id')),
                    ])
                    ->afterValidation(function (array $state, callable $set, \Filament\Resources\Pages\Page $livewire, callable $get) {
                        try {
                            Log::info('Project validation state:', $state);

                            $projectId = $get('project_id');
                            $isEditing = !empty($projectId);

                            if ($isEditing) {
                                $project = Project::find($projectId);
                                if (!$project) {
                                    throw new \Exception('Project not found for editing.');
                                }

                                $project->update([
                                    'name' => $state['name'],
                                    'location' => $state['location'],
                                    'poc' => $state['poc'],
                                    'description' => $state['description'] ?? null,
                                    'type' => $state['type'],
                                    'number_of_units' => $state['number_of_units'] ?? null,
                                ]);

                                Log::info('Project updated successfully with ID: ' . $project->id);
                            } else {
                                $project = Project::create([
                                    'name' => $state['name'],
                                    'location' => $state['location'],
                                    'poc' => $state['poc'],
                                    'description' => $state['description'] ?? null,
                                    'type' => $state['type'],
                                    'number_of_units' => $state['number_of_units'] ?? null,
                                ]);

                                Log::info('Project created successfully with ID: ' . $project->id);
                            }

                            if (!empty($state['documents'])) {
                                $existingDocIds = $isEditing ? $project->documents()->pluck('id')->toArray() : [];
                                $newDocIds = [];

                                foreach ($state['documents'] as $index => $document) {
                                    Log::info('Processing document:', ['index' => $index, 'document' => $document]);

                                    if (empty($document['name'])) {
                                        $livewire->addError("documents.{$index}.name", 'Document name is required.');
                                        throw new \Exception('Document name is required.');
                                    }

                                    if (empty($document['file_path'])) {
                                        $livewire->addError("documents.{$index}.file_path", 'File upload is required for each project document.');
                                        throw new \Exception('File upload is required for each project document.');
                                    }

                                    $filePath = null;
                                    if (is_array($document['file_path'])) {
                                        $firstFile = reset($document['file_path']);
                                        if ($firstFile instanceof TemporaryUploadedFile) {
                                            $filePath = $firstFile->store('project-documents', 'public');
                                        } else {
                                            $filePath = $firstFile;
                                        }
                                    } else {
                                        $filePath = $document['file_path'];
                                    }

                                    if (empty($filePath)) {
                                        Log::error('Empty file path after processing:', ['document' => $document]);
                                        $livewire->addError("documents.{$index}.file_path", 'File upload is required for each project document.');
                                        throw new \Exception('File upload is required for each project document.');
                                    }

                                    Log::info('Creating/updating document with file path:', ['file_path' => $filePath]);

                                    if (isset($document['id']) && $isEditing) {
                                        $existingDoc = $project->documents()->find($document['id']);
                                        if ($existingDoc) {
                                            $existingDoc->update([
                                                'name' => $document['name'],
                                                'file_path' => $filePath,
                                            ]);
                                            $newDocIds[] = $existingDoc->id;
                                        } else {
                                            $newDoc = $project->documents()->create([
                                                'name' => $document['name'],
                                                'file_path' => $filePath,
                                                'project_id' => $project->id,
                                            ]);
                                            $newDocIds[] = $newDoc->id;
                                        }
                                    } else {
                                        $newDoc = $project->documents()->create([
                                            'name' => $document['name'],
                                            'file_path' => $filePath,
                                            'project_id' => $project->id,
                                        ]);
                                        $newDocIds[] = $newDoc->id;
                                    }
                                }

                                if ($isEditing) {
                                    $docsToDelete = array_diff($existingDocIds, $newDocIds);
                                    if (!empty($docsToDelete)) {
                                        $project->documents()->whereIn('id', $docsToDelete)->delete();
                                    }
                                }
                            }

                            $set('project_id', $project->id);
                        } catch (\Exception $e) {
                            Log::error('Failed to save project: ' . $e->getMessage());
                            $livewire->addError('name', 'Failed to save project: ' . $e->getMessage());
                            throw $e;
                        }
                    }),
                Step::make('Project Clients')
                    ->schema([
                        Hidden::make('project_id'),
                        Repeater::make('projectClients')
                            ->label('Project Clients')
                            ->reactive()
                            ->minItems(0)
                            ->defaultItems(1)
                            ->afterStateUpdated(function ($state) {
                                Log::info('ProjectClients state:', ['state' => $state]);
                            })
                            ->schema(function (callable $get) {
                                $projectType = $get('type');
                                return [
                                    Section::make('Client Info')
                                        ->columnSpanFull()
                                        ->schema([
                                            Grid::make(3)->schema([
                                                Select::make('client_id')
                                                    ->label('Client')
                                                    ->options(function () {
                                                        return Client::with('user')->get()->mapWithKeys(fn($client) => [
                                                            $client->id => $client->user->name,
                                                        ]);
                                                    })
                                                    ->nullable()
                                                    ->searchable()
                                                    ->preload()
                                                    ->required()
                                                    ->placeholder('Select or create a client')
                                                    ->reactive()
                                                    ->afterStateUpdated(function ($state, callable $get, callable $set) {
                                                        $client = Client::find($state);
                                                        if (!$client) {
                                                            return;
                                                        }
                                                        $set('email', $client->user->email);
                                                        $set('phone', $client->phone);
                                                        $set('national_id', $client->national_id);
                                                        $set('address', $client->address);
                                                        $set('passport_number', $client->passport_number);
                                                        $set('nationality', $client->nationality);
                                                        $set('kra_pin', $client->kra_pin);
                                                        $set('identification_image', $client->identification_image ? [$client->identification_image] : null);
                                                    })
                                                    ->createOptionForm([
                                                        Grid::make(2)->schema([
                                                            TextInput::make('user.name')
                                                                ->label('Name')
                                                                ->required()
                                                                ->maxLength(255),
                                                            TextInput::make('user.email')
                                                                ->label('Email')
                                                                ->nullable()
                                                                ->maxLength(255)
                                                                ->email(),
                                                            TextInput::make('phone')
                                                                ->required()
                                                                ->tel()
                                                                ->maxLength(20),
                                                            TextInput::make('national_id')
                                                                ->label('National ID')
                                                                ->required()
                                                                ->maxLength(50),
                                                            Textarea::make('address')
                                                                ->required()
                                                                ->rows(2)
                                                                ->maxLength(500),
                                                            TextInput::make('passport_number')
                                                                ->label('Passport Number')
                                                                ->nullable()
                                                                ->maxLength(50),
                                                            TextInput::make('kra_pin')
                                                                ->label('KRA PIN')
                                                                ->nullable()
                                                                ->maxLength(20),
                                                            TextInput::make('nationality')
                                                                ->nullable()
                                                                ->maxLength(20),
                                                            FileUpload::make('identification_image')
                                                                ->disk('public')
                                                                ->directory('client-identifications')
                                                                ->image()
                                                                ->acceptedFileTypes(['image/jpeg', 'image/png'])
                                                                ->maxSize(5120)
                                                                ->multiple(false)
                                                                ->afterStateUpdated(function ($state) {
                                                                    Log::info('FileUpload state updated:', ['file_path' => $state]);
                                                                }),
                                                        ])
                                                    ])
                                                    ->createOptionUsing(function (array $data, \Filament\Resources\Pages\Page $livewire) {
                                                        try {
                                                            $user = User::create([
                                                                'name' => $data['user']['name'],
                                                                'email' => $data['user']['email'],
                                                            ]);
                                                            $user->assignRole('client');

                                                            $client = Client::create([
                                                                'user_id' => $user->id,
                                                                'phone' => $data['phone'],
                                                                'national_id' => $data['national_id'],
                                                                'nationality' => $data['nationality'],
                                                                'address' => $data['address'],
                                                                'kra_pin' => $data['kra_pin'],
                                                                'passport_number' => $data['passport_number'],
                                                                'identification_image' => $data['identification_image'],
                                                            ]);
                                                            Log::info('Client created:', ['client_id' => $client->getKey()]);
                                                            return $client->getKey();
                                                        } catch (\Exception $e) {
                                                            Log::error('Failed to create client: ' . $e->getMessage());
                                                            $livewire->addError('client_id', 'Failed to create client: ' . $e->getMessage());
                                                            return null;
                                                        }
                                                    }),
                                                TextInput::make('phone')
                                                    ->label('Phone')
                                                    ->nullable()
                                                    ->tel()
                                                    ->maxLength(20),
                                                TextInput::make('email')
                                                    ->label('Email')
                                                    ->email()
                                                    ->nullable()
                                                    ->maxLength(255),
                                                // TextInput::make('national_id')->label('National ID')->nullable(),
                                                // TextInput::make('address')->label('Address')->nullable(),
                                                // TextInput::make('nationality')->label('Nationality')->nullable(),
                                                // TextInput::make('passport_number')->label('Passport Number')->nullable()->maxLength(50),
                                                // TextInput::make('kra_pin')->label('KRA PIN')->nullable()->maxLength(20),
                                            ]),
                                            // Grid::make(1)->schema([
                                            //     FileUpload::make('identification_image')
                                            //         ->label('Identification Image')
                                            //         ->disk('public')
                                            //         ->directory('client-identifications')
                                            //         ->acceptedFileTypes(['image/jpeg', 'image/png'])
                                            //         ->image()
                                            //         ->maxSize(5120)
                                            //         ->preserveFilenames()
                                            //         ->visibility('public')
                                            //         ->enableDownload()
                                            //         ->enableOpen()
                                            //         ->nullable()
                                            //         ->dehydrateStateUsing(function ($state) {
                                            //             return is_array($state) ? reset($state) : $state;
                                            //         })
                                            //         ->afterStateHydrated(function ($state, $set) {
                                            //             if (is_string($state)) {
                                            //                 $set('identification_image', [$state]);
                                            //             }
                                            //             Log::info('Identification image hydrated:', [
                                            //                 'state' => $state,
                                            //                 'type' => gettype($state),
                                            //                 'new_state' => is_string($state) ? [$state] : $state,
                                            //                 'url' => $state ? Storage::disk('public')->url(is_array($state) ? reset($state) : $state) : null,
                                            //             ]);
                                            //         })
                                            //         ->afterStateUpdated(function ($state, $set) {
                                            //             Log::info('Identification image state updated:', [
                                            //                 'state' => $state,
                                            //                 'type' => gettype($state),
                                            //                 'is_array' => is_array($state),
                                            //                 'url' => $state ? Storage::disk('public')->url(is_array($state) ? reset($state) : $state) : null,
                                            //             ]);
                                            //             if (is_string($state)) {
                                            //                 $set('identification_image', [$state]);
                                            //             }
                                            //         })
                                            // ]),
                                        ]),
                                    Section::make('Apartment Details')
                                        ->columnSpanFull()
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('apartmentDetail.floor')
                                                    ->label('Floor')
                                                    ->numeric()
                                                    ->required(),
                                                TextInput::make('apartmentDetail.flat_name')
                                                    ->label('Unit Name')
                                                    ->required()
                                                    ->placeholder('e.g., FLAT A67')
                                                    ->maxLength(50),
                                                Select::make('apartmentDetail.unit_type')
                                                    ->label('Unit Type')
                                                    ->options([
                                                        '1-bedroom' => '1-bedroom',
                                                        '2-bedroom' => '2-bedroom',
                                                        '3-bedroom' => '3-bedroom',
                                                        '4-bedroom' => '4-bedroom',
                                                    ])
                                                    ->required(),
                                                TextInput::make('apartmentDetail.owner2')
                                                    ->label('Owner 2')
                                                    ->nullable()
                                                    ->maxLength(255),
                                                TextInput::make('apartmentDetail.amount_charged')
                                                    ->label('Amount Charged')
                                                    ->numeric()
                                                    ->required()
                                                    ->reactive(),
                                                Hidden::make('apartmentDetail.project_id')
                                                    ->default($get('project_id')),
                                            ]),
                                        ])
                                        ->visible($projectType === 'apartment'),
                                    Section::make('Subdivision Details')
                                        ->columnSpanFull()
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('subdivisionDetail.plot_number')
                                                    ->label('Plot Number')
                                                    ->required()
                                                    ->placeholder('e.g., PLOT 45')
                                                    ->maxLength(50),
                                                TextInput::make('subdivisionDetail.size')
                                                    ->label('Size')
                                                    ->maxLength(255)
                                                    ->required(),
                                                TextInput::make('subdivisionDetail.owner2')
                                                    ->label('Owner 2')
                                                    ->nullable()
                                                    ->maxLength(255),
                                                TextInput::make('subdivisionDetail.amount_charged')
                                                    ->label('Amount Charged')
                                                    ->numeric()
                                                    ->required()
                                                    ->minValue(0),
                                                Hidden::make('subdivisionDetail.project_id')
                                                    ->default($get('project_id')),
                                            ]),
                                        ])
                                        ->visible($projectType === 'sub_division'),
                                    Repeater::make('documents')
                                        ->label('Client Documents')
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('name')
                                                    ->required()
                                                    ->maxLength(255),
                                                FileUpload::make('file_path')
                                                    ->disk('public')
                                                    ->directory('project-client-documents')
                                                    ->required()
                                                    ->acceptedFileTypes(['application/pdf', 'image/*'])
                                                    ->maxSize(5120),
                                            ]),
                                        ])
                                        ->collapsible()
                                        ->defaultItems(0),
                                ];
                            })
                            ->collapsible()
                            ->itemLabel(fn(array $state): ?string => isset($state['client_id']) ? Client::find($state['client_id'])?->user->name : null),
                    ])
                    ->visible(fn($livewire) => $livewire instanceof \App\Filament\Resources\ProjectResource\Pages\EditProject),
            ])
                ->columnSpanFull(),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('location')
                    ->sortable(),
                Tables\Columns\TextColumn::make('type')
                    ->badge()
                    ->formatStateUsing(fn($state) => ucfirst(str_replace('_', ' ', $state))),
                Tables\Columns\TextColumn::make('number_of_units')
                    ->label('Units')
                    ->sortable(),
            ])
            ->filters([])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
                Tables\Actions\Action::make('add_clients')
                    ->label('Add Clients')
                    ->form(function ($record) {
                        return [
                            Repeater::make('projectClients')
                                ->label('Project Clients')
                                ->schema([
                                    Section::make('Client Info')->schema([
                                        Grid::make(3)->schema([
                                            Select::make('client_id')
                                                ->label('Client')
                                                ->options(function () {
                                                    return Client::with('user')->get()->mapWithKeys(fn($client) => [
                                                        $client->id => $client->user->name,
                                                    ]);
                                                })
                                                ->nullable()
                                                ->searchable()
                                                ->preload()
                                                ->required()
                                                ->placeholder('Select or create a client')
                                                ->reactive()
                                                ->afterStateUpdated(function ($state, callable $get, callable $set) {
                                                    $client = Client::find($state);
                                                    if (!$client) {
                                                        return;
                                                    }
                                                    $set('email', $client->user->email);
                                                    $set('phone', $client->phone);
                                                    $set('national_id', $client->national_id);
                                                    $set('address', $client->address);
                                                    $set('passport_number', $client->passport_number);
                                                    $set('nationality', $client->nationality);
                                                    $set('kra_pin', $client->kra_pin);
                                                    $set('identification_image', $client->identification_image ? [$client->identification_image] : null);
                                                })
                                                ->createOptionForm([
                                                    Grid::make(2)->schema([
                                                        TextInput::make('user.name')
                                                            ->label('Name')
                                                            ->required()
                                                            ->maxLength(255),
                                                        TextInput::make('user.email')
                                                            ->label('Email')
                                                            ->nullable()
                                                            ->maxLength(255)
                                                            ->email(),
                                                        TextInput::make('phone')
                                                            ->required()
                                                            ->tel()
                                                            ->maxLength(20),
                                                        TextInput::make('national_id')
                                                            ->label('National ID')
                                                            ->required()
                                                            ->maxLength(50),
                                                        Textarea::make('address')
                                                            ->required()
                                                            ->rows(2)
                                                            ->maxLength(500),
                                                        TextInput::make('passport_number')
                                                            ->label('Passport Number')
                                                            ->nullable()
                                                            ->maxLength(50),
                                                        TextInput::make('kra_pin')
                                                            ->label('KRA PIN')
                                                            ->nullable()
                                                            ->maxLength(20),
                                                        TextInput::make('nationality')
                                                            ->nullable()
                                                            ->maxLength(20),
                                                        FileUpload::make('identification_image')
                                                            ->disk('public')
                                                            ->directory('client-identifications')
                                                            ->image()
                                                            ->acceptedFileTypes(['image/jpeg', 'image/png'])
                                                            ->maxSize(5120)
                                                            ->multiple(false)
                                                            ->afterStateUpdated(function ($state) {
                                                                Log::info('FileUpload state updated:', ['file_path' => $state]);
                                                            }),
                                                    ])
                                                ])
                                                ->createOptionUsing(function (array $data, \Filament\Resources\Pages\Page $livewire) {
                                                    try {
                                                        $user = User::create([
                                                            'name' => $data['user']['name'],
                                                            'email' => $data['user']['email'],
                                                        ]);
                                                        $user->assignRole('client');

                                                        $identificationImage = null;
                                                        if (!empty($data['identification_image'])) {
                                                            if ($data['identification_image'] instanceof TemporaryUploadedFile) {
                                                                if (!$data['identification_image']->exists()) {
                                                                    Log::error('Temporary file does not exist for identification_image:', ['path' => $data['identification_image']->getPathname()]);
                                                                    throw new \Exception('Temporary file for identification image does not exist.');
                                                                }
                                                                $identificationImage = $data['identification_image']->store('client-identifications', 'public');
                                                                Log::info('Identification image stored successfully:', ['path' => $identificationImage]);
                                                            } else {
                                                                $identificationImage = is_array($data['identification_image']) ? reset($data['identification_image']) : $data['identification_image'];
                                                            }
                                                        }
                                                        $client = Client::create([
                                                            'user_id' => $user->id,
                                                            'phone' => $data['phone'],
                                                            'national_id' => $data['national_id'],
                                                            'address' => $data['address'],
                                                            'kra_pin' => $data['kra_pin'],
                                                            'passport_number' => $data['passport_number'],
                                                            'nationality' => $data['nationality'],
                                                            'identification_image' => $identificationImage,
                                                        ]);
                                                        Log::info('Client created:', ['client_id' => $client->getKey()]);
                                                        return $client->getKey();
                                                    } catch (\Exception $e) {
                                                        Log::error('Failed to create client: ' . $e->getMessage());
                                                        $livewire->addError('client_id', 'Failed to create client: ' . $e->getMessage());
                                                        return null;
                                                    }
                                                }),
                                            TextInput::make('phone')
                                                ->label('Phone')
                                                ->nullable()
                                                ->tel()
                                                ->maxLength(20),
                                            TextInput::make('email')
                                                ->label('Email')
                                                ->nullable()
                                                ->email()
                                                ->maxLength(255),
                                            // TextInput::make('national_id')->label('National ID')->nullable()->maxLength(50),
                                            // Textarea::make('address')->label('Address')->nullable()->rows(2)->maxLength(500),
                                            // TextInput::make('nationality')->nullable()->maxLength(255),
                                            // TextInput::make('passport_number')->label('Passport Number')->nullable()->maxLength(50),
                                            // TextInput::make('kra_pin')->label('KRA PIN')->nullable()->maxLength(20),
                                        ]),
                                        // Grid::make(1)->schema([
                                        //     FileUpload::make('identification_image')
                                        //         ->disk('public')
                                        //         ->directory('client-identifications')
                                        //         ->image()
                                        //         ->maxSize(2048)
                                        //         ->acceptedFileTypes(['image/jpeg', 'image/png'])
                                        //         ->nullable()
                                        //         ->dehydrateStateUsing(function ($state) {
                                        //             return is_array($state) ? reset($state) : $state;
                                        //         })
                                        //         ->afterStateHydrated(function ($state, $set) {
                                        //             if (is_string($state)) {
                                        //                 $set('identification_image', [$state]);
                                        //             }
                                        //             Log::info('Identification image hydrated:', [
                                        //                 'state' => $state,
                                        //                 'type' => gettype($state),
                                        //                 'new_state' => is_string($state) ? [$state] : $state,
                                        //                 'url' => $state ? Storage::disk('public')->url(is_array($state) ? reset($state) : $state) : null,
                                        //             ]);
                                        //         })
                                        //         ->afterStateUpdated(function ($state, $set) {
                                        //             Log::info('Identification image state updated:', [
                                        //                 'state' => $state,
                                        //                 'type' => gettype($state),
                                        //                 'is_array' => is_array($state),
                                        //                 'url' => $state ? Storage::disk('public')->url(is_array($state) ? reset($state) : $state) : null,
                                        //             ]);
                                        //             if (is_string($state)) {
                                        //                 $set('identification_image', [$state]);
                                        //             }
                                        //         }),
                                        // ]),
                                    ]),
                                    Section::make('Apartment Details')
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('apartmentDetail.floor')
                                                    ->label('Floor')
                                                    ->numeric()
                                                    ->required(),
                                                TextInput::make('apartmentDetail.flat_name')
                                                    ->label('Unit Name')
                                                    ->required()
                                                    ->placeholder('e.g., FLAT A67')
                                                    ->maxLength(50),
                                                Select::make('apartmentDetail.unit_type')
                                                    ->label('Unit Type')
                                                    ->options([
                                                        '1-bedroom' => '1-bedroom',
                                                        '2-bedroom' => '2-bedroom',
                                                        '3-bedroom' => '3-bedroom',
                                                        '4-bedroom' => '4-bedroom',
                                                    ])
                                                    ->required(),
                                                TextInput::make('apartmentDetail.amount_charged')
                                                    ->label('Amount Charged')
                                                    ->numeric()
                                                    ->required()
                                                    ->minValue(0),
                                                TextInput::make('apartmentDetail.owner2')
                                                    ->label('Owner 2')
                                                    ->nullable()
                                                    ->maxLength(255),
                                            ]),
                                        ])
                                        ->visible(fn() => $record->type === 'apartment'),
                                    Section::make('Subdivision Details')
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('subdivisionDetail.plot_number')
                                                    ->label('Plot Number')
                                                    ->required()
                                                    ->placeholder('e.g., PLOT 45')
                                                    ->maxLength(50),
                                                TextInput::make('subdivisionDetail.size')
                                                    ->label('Size in Acres')
                                                    ->numeric()
                                                    ->required()
                                                    ->minValue(0),
                                                TextInput::make('subdivisionDetail.amount_charged')
                                                    ->label('Amount Charged')
                                                    ->numeric()
                                                    ->required()
                                                    ->minValue(0),
                                                TextInput::make('subdivisionDetail.owner2')
                                                    ->label('Owner 2')
                                                    ->nullable()
                                                    ->maxLength(255),
                                            ]),
                                        ])
                                        ->visible(fn() => $record->type === 'sub_division'),
                                    Repeater::make('documents')
                                        ->label('Client Documents')
                                        ->schema([
                                            Grid::make(2)->schema([
                                                TextInput::make('name')
                                                    ->required()
                                                    ->maxLength(255),
                                                FileUpload::make('file_path')
                                                    ->disk('public')
                                                    ->directory('project-client-documents')
                                                    ->required()
                                                    ->acceptedFileTypes(['application/pdf', 'image/*'])
                                                    ->maxSize(5120),
                                            ]),
                                        ])
                                        ->collapsible()
                                        ->defaultItems(0),
                                ])
                                ->collapsible()
                                ->defaultItems(1)
                                ->itemLabel(fn(array $state): ?string => isset($state['client_id']) ? Client::find($state['client_id'])?->user->name : null),
                        ];
                    })
                    ->action(function (array $data, $record) {
                        return DB::transaction(function () use ($data, $record) {
                            try {
                                foreach ($data['projectClients'] as $index => $clientData) {
                                    $identificationImage = null;
                                    if (!empty($clientData['identification_image'])) {
                                        Log::info('Processing identification_image for client:', ['index' => $index, 'identification_image' => $clientData['identification_image']]);
                                        if (is_array($clientData['identification_image'])) {
                                            $firstFile = reset($clientData['identification_image']);
                                            if ($firstFile instanceof TemporaryUploadedFile) {
                                                if (!$firstFile->exists()) {
                                                    Log::error('Temporary file does not exist for identification_image:', ['path' => $firstFile->getPathname()]);
                                                    throw new \Exception('Temporary file for identification image does not exist.');
                                                }
                                                $identificationImage = $firstFile->store('client-identifications', 'public');
                                                Log::info('Identification image stored successfully:', ['path' => $identificationImage]);
                                            } else {
                                                $identificationImage = $firstFile;
                                            }
                                        } else {
                                            $identificationImage = $clientData['identification_image'];
                                        }
                                    }

                                    $client = null;
                                    if (isset($clientData['client_id']) && $clientData['client_id']) {
                                        $client = Client::find($clientData['client_id']);
                                        if ($client) {
                                            $client->update([
                                                'phone' => $clientData['phone'] ?? $client->phone,
                                                'national_id' => $clientData['national_id'] ?? $client->national_id,
                                                'address' => $clientData['address'] ?? $client->address,
                                                'nationality' => $clientData['nationality'] ?? $client->nationality,
                                                'passport_number' => $clientData['passport_number'] ?? $client->passport_number,
                                                'kra_pin' => $clientData['kra_pin'] ?? $client->kra_pin,
                                                'identification_image' => $identificationImage ?? $client->identification_image,
                                            ]);

                                            if (isset($clientData['email']) && $client->user) {
                                                $client->user->update([
                                                    'email' => $clientData['email'] ?? $client->user->email,
                                                ]);
                                            }

                                            Log::info('Client updated:', [
                                                'client_id' => $client->id,
                                                'updated_email' => $clientData['email'] ?? null,
                                                'data' => [
                                                    'phone' => $clientData['phone'],
                                                ],
                                            ]);
                                        } else {
                                            Log::warning('Client not found for ID:', ['client_id' => $clientData['client_id']]);
                                            throw new \Exception('Client not found for ID: ' . $clientData['client_id']);
                                        }
                                    } else {
                                        $user = User::create([
                                            'name' => $clientData['user']['name'] ?? 'Unknown',
                                            'email' => $clientData['user']['email'] ?? null,
                                        ]);
                                        $user->assignRole('client');

                                        $client = Client::create([
                                            'user_id' => $user->id,
                                            'phone' => $clientData['phone'] ?? null,
                                            'national_id' => $clientData['national_id'] ?? null,
                                            'address' => $clientData['address'] ?? null,
                                            'nationality' => $clientData['nationality'] ?? null,
                                            'passport_number' => $clientData['passport_number'] ?? null,
                                            'kra_pin' => $clientData['kra_pin'] ?? null,
                                            'identification_image' => $identificationImage,
                                        ]);
                                        Log::info('Client created:', ['client_id' => $client->id]);
                                        $clientData['client_id'] = $client->id;
                                    }

                                    $projectClient = $record->projectClients()->create([
                                        'client_id' => $client->id,
                                    ]);

                                    if ($record->type === 'apartment' && isset($clientData['apartmentDetail'])) {
                                        $projectClient->apartmentDetail()->create([
                                            'project_id' => $record->id,
                                            'project_client_id' => $projectClient->id,
                                            'floor' => $clientData['apartmentDetail']['floor'],
                                            'flat_name' => $clientData['apartmentDetail']['flat_name'],
                                            'unit_type' => $clientData['apartmentDetail']['unit_type'],
                                            'owner2' => $clientData['apartmentDetail']['owner2'] ?? null,
                                            'amount_charged' => $clientData['apartmentDetail']['amount_charged'],
                                        ]);
                                    } elseif ($record->type === 'sub_division' && isset($clientData['subdivisionDetail'])) {
                                        $projectClient->subdivisionDetail()->create([
                                            'project_id' => $record->id,
                                            'project_client_id' => $projectClient->id,
                                            'plot_number' => $clientData['subdivisionDetail']['plot_number'],
                                            'size' => $clientData['subdivisionDetail']['size'],
                                            'owner2' => $clientData['subdivisionDetail']['owner2'] ?? null,
                                            'amount_charged' => $clientData['subdivisionDetail']['amount_charged'],
                                        ]);
                                    }

                                    if (!empty($clientData['documents'])) {
                                        foreach ($clientData['documents'] as $docIndex => $document) {
                                            Log::info('Processing projectClientDocument:', ['client_index' => $index, 'doc_index' => $docIndex, 'document' => $document]);
                                            $filePath = null;
                                            if (is_array($document['file_path'])) {
                                                $firstFile = reset($document['file_path']);
                                                if ($firstFile instanceof TemporaryUploadedFile) {
                                                    if (!$firstFile->exists()) {
                                                        Log::error('Temporary file does not exist for projectClientDocument:', ['path' => $firstFile->getPathname()]);
                                                        throw new \Exception('Temporary file for project client document does not exist.');
                                                    }
                                                    $filePath = $firstFile->store('project-client-documents', 'public');
                                                    Log::info('Project client document stored successfully:', ['path' => $filePath]);
                                                } else {
                                                    $filePath = $firstFile;
                                                }
                                            } else {
                                                $filePath = $document['file_path'];
                                            }

                                            if ($filePath === null) {
                                                continue;
                                            }

                                            $projectClient->documents()->create([
                                                'name' => $document['name'],
                                                'file_path' => $filePath,
                                            ]);
                                        }
                                    }
                                }

                                Notification::make()
                                    ->title('Success')
                                    ->body('Clients added successfully.')
                                    ->success()
                                    ->send();
                            } catch (\Exception $e) {
                                Log::error('Failed to add clients: ' . $e->getMessage(), ['exception' => $e]);
                                Notification::make()
                                    ->title('Error')
                                    ->body('Failed to add clients: ' . $e->getMessage())
                                    ->danger()
                                    ->send();
                                throw $e;
                            }
                        });
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $project = $this->record ?? null;

        if ($project) {
            $data['project_id'] = $project->id;

            $data['documents'] = $project->documents()->get()->map(function ($doc) {
                return [
                    'id' => $doc->id,
                    'name' => $doc->name,
                    'file_path' => $doc->file_path,
                ];
            })->toArray();

            $data['projectClients'] = $project->projectClients()->with([
                'client.user',
                'apartmentDetail',
                'subdivisionDetail',
                'documents',
            ])->get()->map(function ($projectClient) {
                $clientData = [
                    'id' => $projectClient->id,
                    'client_id' => $projectClient->client_id,
                    'passport_number' => $projectClient->client->passport_number,
                    'national_id' => $projectClient->client->national_id,
                    'address' => $projectClient->client->address,
                    'phone' => $projectClient->client->phone,
                    'nationality' => $projectClient->client->nationality,
                    'kra_pin' => $projectClient->client->kra_pin,
                    'identification_image' => $projectClient->client->identification_image,
                    'documents' => $projectClient->documents->map(function ($doc) {
                        return [
                            'id' => $doc->id,
                            'name' => $doc->name,
                            'file_path' => $doc->file_path,
                        ];
                    })->toArray(),
                ];

                if ($projectClient->apartmentDetail) {
                    $clientData['apartmentDetail'] = [
                        'id' => $projectClient->apartmentDetail->id,
                        'floor' => $projectClient->apartmentDetail->floor,
                        'flat_name' => $projectClient->apartmentDetail->flat_name,
                        'unit_type' => $projectClient->apartmentDetail->unit_type,
                        'owner2' => $projectClient->apartmentDetail->owner2,
                        'amount_charged' => $projectClient->apartmentDetail->amount_charged,
                        'project_id' => $projectClient->apartmentDetail->project_id,
                    ];
                } elseif ($projectClient->subdivisionDetail) {
                    $clientData['subdivisionDetail'] = [
                        'id' => $projectClient->subdivisionDetail->id,
                        'plot_number' => $projectClient->subdivisionDetail->plot_number,
                        'size' => $projectClient->subdivisionDetail->size,
                        'owner2' => $projectClient->subdivisionDetail->owner2,
                        'amount_charged' => $projectClient->subdivisionDetail->amount_charged,
                        'project_id' => $projectClient->subdivisionDetail->project_id,
                    ];
                }

                return $clientData;
            })->toArray();
        }

        return $data;
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProjects::route('/'),
            'create' => Pages\CreateProject::route('/create'),
            'view' => Pages\ViewProject::route('/{record}'),
            'edit' => Pages\EditProject::route('/{record}/edit'),
        ];
    }
}