<?php

namespace App\Filament\Resources\PaymentResource\Pages;

use App\Filament\Resources\PaymentResource;
use App\Models\Payment;
use App\Models\Salary;
use App\Models\OfficeUtility;
use App\Models\LitigationPayment;
use App\Models\ProjectClientPayment;
use App\Models\ConveyancingPayment;
use App\Models\ProjectClient;
use App\Models\Expense;
use App\Models\LitigationExpense;
use App\Models\ConveyancingExpense;
use App\Models\ProjectExpense;
use Filament\Resources\Pages\CreateRecord;
use Filament\Actions;
use Filament\Notifications\Notification;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Response;

class CreatePointOfSale extends CreateRecord
{
    protected static string $resource = PaymentResource::class;
   
    protected static string $view = 'filament.resources.payment-resource.pages.create-point-of-sale';

    public function getTitle(): string
    {
        return 'Point of Sale - Transaction Entry';
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('back')
                ->label('Back to Payments')
                ->url($this->getResource()::getUrl('index'))
                ->color('gray')
                ->icon('heroicon-o-arrow-left'),
        ];
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('index');
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // Handle the type field logic for incoming transactions
        if ($data['direction'] === 'incoming') {
            if ($data['payable_type'] === 'App\Models\Litigation') {
                $data['type'] = 'legal_fees';
            } elseif ($data['payable_type'] === 'App\Models\Conveyancing') {
                $data['type'] = 'legal_fees';
            }
        }

        // Set account_id to null if not bank transfer
        if ($data['payment_method'] !== 'bank') {
            $data['account_id'] = null;
        }

        // Handle defaults for outgoing payments if payable_id is null
        if ($data['direction'] === 'outgoing' && empty($data['payable_id'])) {
            $currentPeriod = date('Y-m');
            if ($data['payable_type'] === 'App\Models\Salary') {
                if (empty($data['client_id'])) {
                    throw new \Exception('Employee is required for Salary payment');
                }
                $salary = Salary::create([
                    'employee_id' => $data['client_id'],
                    'amount' => $data['amount_paid'],
                    'period' => $currentPeriod,
                ]);
                $data['payable_id'] = $salary->id;
            } elseif ($data['payable_type'] === 'App\Models\OfficeUtility') {
                $officeUtility = OfficeUtility::create([
                    'type' => 'General Office Expense',
                    'amount' => $data['amount_paid'],
                    'period' => $currentPeriod,
                ]);
                $data['payable_id'] = $officeUtility->id;
            } elseif ($data['payable_type'] === 'App\Models\Expense') {
                if (empty($data['expense_item_id']) || empty($data['expensable_type']) || empty($data['expensable_id'])) {
                    throw new \Exception('All expense details are required');
                }
                // dd($data);
                $expense = Expense::create([
                    'expense_item_id' => $data['expense_item_id'],
                    'amount_paid' => $data['amount_paid'],
                    'balance' => 0,  // Assuming full payment; adjust if needed
                    'payment_method' => $data['payment_method'],
                    'status' => 'completed',
                    'client_id' => $data['paid_by'] ?? null,  
                    'expensable_id' => $data['expensable_id'],
                    'expensable_type' => $data['expensable_type'],
                ]);
                $data['payable_id'] = $expense->id;
            }
        }

        // Set default status to completed
        $data['status'] = 'completed';

        return $data;
    }

    protected function afterCreate(): void
    {
        $payment = $this->record;
        $data = $this->data;

        if ($payment->direction === 'incoming' && $payment->client_id && $payment->payable_id) {
            switch ($payment->payable_type) {
                case 'App\Models\Litigation':
                    LitigationPayment::create([
                        'litigation_id' => $payment->payable_id,
                        'client_id' => $payment->client_id,
                        'amount_paid' => $payment->amount_paid,
                        'payment_method' => $payment->payment_method,
                        'transaction_id' => $payment->transaction_id,
                        'status' => $payment->status,
                        'type' => $payment->type ?? 'legal_fees',
                        'payment_id' => $payment->id,
                    ]);
                    break;
                case 'App\Models\Project':
                    $projectClient = ProjectClient::where('project_id', $payment->payable_id)
                        ->where('client_id', $payment->client_id)
                        ->first();
                    if ($projectClient) {
                        ProjectClientPayment::create([
                            'project_client_id' => $projectClient->id,
                            'amount_paid' => $payment->amount_paid,
                            'payment_method' => $payment->payment_method,
                            'transaction_id' => $payment->transaction_id,
                            'status' => $payment->status,
                            'payment_id' => $payment->id,
                        ]);
                    }
                    break;
                case 'App\Models\Conveyancing':
                    ConveyancingPayment::create([
                        'conveyancing_id' => $payment->payable_id,
                        'client_id' => $payment->client_id,
                        'amount_paid' => $payment->amount_paid,
                        'payment_method' => $payment->payment_method,
                        'transaction_id' => $payment->transaction_id,
                        'status' => $payment->status,
                        'type' => $payment->type,
                        'payment_id' => $payment->id,
                        'paid_by' => $data['paid_by'] ?? null,
                    ]);
                    break;
            }
        } else if ($payment->direction === 'outgoing' && $payment->payable_type === 'App\Models\Expense') {
            switch ($data['expensable_type']) {
                case 'App\Models\Litigation':
                    LitigationExpense::create([
                        'litigation_id' => $data['expensable_id'],
                        'expense_id' => $payment->payable_id,
                    ]);
                    break;
                case 'App\Models\Conveyancing':
                    ConveyancingExpense::create([
                        'conveyancing_id' => $data['expensable_id'],
                        'expense_id' => $payment->payable_id,
                    ]);
                    break;
                case 'App\Models\Project':
                    ProjectExpense::create([
                        'project_id' => $data['expensable_id'],
                        'expense_id' => $payment->payable_id,
                    ]);
                    break;
            }
        }

        // Generate and immediately show receipt
        $this->generateAndShowReceipt($payment);
    }

    private function generateAndShowReceipt(Payment $payment): void
    {
        try {
            // Load the payment with all required relationships
            $payment->load(['client.user', 'payable', 'account']);
           
            // Generate PDF
            $pdf = Pdf::loadView('reports.payment-receipt', ['payment' => $payment])
                ->setPaper('a4')
                ->setOptions(['defaultFont' => 'sans-serif']);
           
            $fileName = 'payment_receipt_' . $payment->id . '_' . now()->format('Y-m-d_H-i-s') . '.pdf';
            $filePath = 'receipts/' . $fileName;
           
            // Store the PDF
            Storage::disk('public')->put($filePath, $pdf->output());
           
            // Create download URL using route
            $downloadUrl = route('receipts.download', $payment->id);
           
            // Show success notification with download button
            Notification::make()
                ->title('Transaction Processed Successfully!')
                ->body("Receipt #{$payment->id} has been generated")
                ->success()
                ->persistent()
                ->actions([
                    \Filament\Notifications\Actions\Action::make('download')
                        ->label('Download Receipt')
                        ->icon('heroicon-o-arrow-down-tray')
                        ->url($downloadUrl)
                        ->openUrlInNewTab()
                        ->button(),
                    \Filament\Notifications\Actions\Action::make('view')
                        ->label('View Receipt')
                        ->icon('heroicon-o-eye')
                        ->url($downloadUrl)
                        ->openUrlInNewTab(),
                ])
                ->send();
           
            // Also store filename in session for additional handling if needed
            session()->flash('pdf_generated', [
                'filename' => $fileName,
                'path' => $filePath,
                'url' => $downloadUrl,
                'payment_id' => $payment->id
            ]);

            // Dispatch Livewire event for frontend handling
            $this->dispatch('payment-processed', [
                'filename' => $fileName,
                'downloadUrl' => $downloadUrl,
                'paymentId' => $payment->id
            ]);
           
        } catch (\Exception $e) {
            Notification::make()
                ->title('Receipt Generation Failed')
                ->body('Transaction was processed but receipt generation failed: ' . $e->getMessage())
                ->warning()
                ->send();
        }
    }

    protected function getCreatedNotification(): ?Notification
    {
        // Return null to prevent default notification since we handle it in generateAndShowReceipt
        return null;
    }

    protected function getViewData(): array
    {
        return [
            'paymentMethods' => [
                'mpesa' => 'M-Pesa',
                'bank' => 'Bank Transfer',
                'cash' => 'Cash',
            ],
        ];
    }
}