<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PaymentResource\Pages;
use App\Models\Payment;
use App\Models\Salary;
use App\Models\OfficeUtility;
use App\Models\Client;
use App\Models\Project;
use App\Models\Conveyancing;
use App\Models\Litigation;
use App\Models\Account;
use App\Models\Employee;
use App\Models\ExpenseItem;
use App\Models\Expense;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Hidden;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Actions\Action;
use Filament\Forms\Get;
use Filament\Forms\Set;

class PaymentResource extends Resource
{
    protected static ?string $model = Payment::class;

    protected static ?string $navigationIcon = 'heroicon-o-arrow-up-circle';

    protected static ?string $navigationLabel = 'Point of Sale';

    protected static ?string $navigationGroup = 'Financial Management';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Transaction Details')
                    ->schema([
                        Select::make('direction')
                            ->label('Transaction Type')
                            ->options([
                                'outgoing' => 'Payment',
                                'incoming' => 'Deposit',
                            ])
                            ->default('outgoing')
                            ->live()
                            ->required()
                            ->afterStateUpdated(function (Set $set) {
                                $set('payable_type', null);
                                $set('payable_id', null);
                                $set('client_id', null);
                                $set('paid_by', null);
                                $set('type', null);
                                $set('account_id', null);
                                $set('amount_paid', null);
                                $set('expense_item_id', null);
                                $set('expensable_type', null);
                                $set('expensable_id', null);
                            }),

                        Select::make('payable_type')
                            ->label('Type')
                            ->options(function (Get $get) {
                                $direction = $get('direction');
                                if ($direction === 'outgoing') {
                                    return [
                                        'App\Models\Salary' => 'Salary',
                                        'App\Models\OfficeUtility' => 'Office Utility',
                                        'App\Models\Expense' => 'Expense to File',
                                    ];
                                } elseif ($direction === 'incoming') {
                                    return [
                                        'App\Models\Project' => 'Project',
                                        'App\Models\Conveyancing' => 'Conveyancing',
                                        'App\Models\Litigation' => 'Litigation',
                                    ];
                                }
                                return [];
                            })
                            ->live()
                            ->afterStateUpdated(function (Set $set) {
                                $set('payable_id', null);
                                $set('client_id', null);
                                $set('paid_by', null);
                                $set('type', null);
                                $set('account_id', null);
                                $set('amount_paid', null);
                                $set('expense_item_id', null);
                                $set('expensable_type', null);
                                $set('expensable_id', null);
                            }),

                        Select::make('client_id')
                            ->label(function (Get $get) {
                                $direction = $get('direction');
                                $type = $get('payable_type');
                                if ($direction === 'outgoing' && $type === 'App\Models\Salary') {
                                    return 'Employee';
                                }
                                return $direction === 'incoming' ? 'Client (Optional)' : 'Client (Optional)';
                            })
                            ->options(function (Get $get) {
                                $direction = $get('direction');
                                $type = $get('payable_type');
                                
                                if ($direction === 'outgoing') {
                                    if ($type === 'App\Models\Salary') {
                                        return Employee::all()->mapWithKeys(function ($employee) {
                                            return [$employee->id => $employee->name ?? 'Employee #' . $employee->id];
                                        });
                                    } else {
                                        return Client::with('user')->get()->mapWithKeys(function ($client) {
                                            $name = $client->user?->name ?? $client->phone ?? 'Client #' . $client->id;
                                            return [$client->id => $name];
                                        });
                                    }
                                } elseif ($direction === 'incoming' && $type) {
                                    $query = Client::with('user');
                                    if ($type === 'App\Models\Project') {
                                        $query->whereHas('projectClients');
                                    } elseif ($type === 'App\Models\Conveyancing') {
                                        $query->whereHas('conveyancings');
                                    } elseif ($type === 'App\Models\Litigation') {
                                        $query->whereHas('litigations');
                                    }
                                    return $query->get()->mapWithKeys(function ($client) {
                                        $name = $client->user?->name ?? $client->phone ?? 'Client #' . $client->id;
                                        return [$client->id => $name];
                                    });
                                }
                                return [];
                            })
                            ->searchable()
                            ->preload()
                            ->live()
                            ->afterStateUpdated(function (Set $set) {
                                $set('payable_id', null);
                                $set('paid_by', null);
                                $set('type', null);
                                $set('account_id', null);
                                $set('amount_paid', null);
                            })
                            ->visible(fn (Get $get) => filled($get('direction')) && filled($get('payable_type')) && !($get('direction') === 'outgoing' && $get('payable_type') === 'App\Models\OfficeUtility') && !($get('direction') === 'outgoing' && $get('payable_type') === 'App\Models\Expense')),

                        Select::make('expense_item_id')
                            ->label('Expense Item')
                            ->options(ExpenseItem::pluck('expense_item', 'id'))
                            ->searchable()
                            ->createOptionForm([
                                TextInput::make('expense_item')
                                    ->label('Expense Item Name')
                                    ->required()
                                    ->maxLength(255)
                                    ->unique(ExpenseItem::class, 'expense_item'),
                            ])
                            ->createOptionUsing(function (array $data): int {
                                return ExpenseItem::create($data)->id;
                            })
                            ->required(fn (Get $get) => $get('payable_type') === 'App\Models\Expense')
                            ->visible(fn (Get $get) => $get('payable_type') === 'App\Models\Expense'),

                        Select::make('expensable_type')
                            ->label('Related To')
                            ->options([
                                'App\Models\Litigation' => 'Litigation',
                                'App\Models\Conveyancing' => 'Conveyancing',
                                'App\Models\Project' => 'Project',
                            ])
                            ->live()
                            ->afterStateUpdated(function (Set $set) {
                                $set('expensable_id', null);
                                $set('paid_by', null);
                                $set('client_id', null);
                            })
                            ->required(fn (Get $get) => $get('payable_type') === 'App\Models\Expense')
                            ->visible(fn (Get $get) => $get('payable_type') === 'App\Models\Expense'),

                        Select::make('expensable_id')
                            ->label('Related Record')
                            ->options(function (Get $get) {
                                $type = $get('expensable_type');
                                if ($type === 'App\Models\Litigation') {
                                    return Litigation::pluck('case_number', 'id');
                                } elseif ($type === 'App\Models\Conveyancing') {
                                    return Conveyancing::with('client.user')->get()->mapWithKeys(function ($convey) {
                                        $clientName = $convey->client?->user?->name ?? 'Unknown';
                                        return [$convey->id => $convey->serial_number . ' - ' . $clientName];
                                    });
                                } elseif ($type === 'App\Models\Project') {
                                    return Project::pluck('name', 'id');
                                }
                                return [];
                            })
                            ->searchable()
                            ->preload()
                            ->live()
                            ->afterStateUpdated(function (Set $set, Get $get, $state) {
                                $type = $get('expensable_type');
                                if ($type === 'App\Models\Conveyancing' && $state) {
                                    $conveyancing = Conveyancing::find($state);
                                    if ($conveyancing) {
                                        $set('client_id', $conveyancing->client_id);
                                    }
                                }
                            })
                            ->required(fn (Get $get) => $get('payable_type') === 'App\Models\Expense')
                            ->visible(fn (Get $get) => $get('payable_type') === 'App\Models\Expense' && filled($get('expensable_type'))),

                        Select::make('paid_by')
                            ->label('Pay Against')
                            ->options(function (Get $get) {
                                $type = $get('expensable_type');
                                $payableId = $get('expensable_id');
                                if ($type === 'App\Models\Conveyancing' && $payableId) {
                                    $conveyancing = Conveyancing::with(['buyer.user', 'seller.user'])->find($payableId);
                                    $options = [];
                                    if ($conveyancing?->buyer) {
                                        $options[$conveyancing->buyer_id] = ($conveyancing->buyer->user?->name ?? 'Buyer') . ' (Buyer)';
                                    }
                                    if ($conveyancing?->seller) {
                                        $options[$conveyancing->seller_id] = ($conveyancing->seller->user?->name ?? 'Seller') . ' (Seller)';
                                    }
                                    return $options;
                                }
                                return [];
                            })
                            ->live()
                            ->afterStateUpdated(function (Set $set, Get $get, $state) {
                                $set('client_id', $state);
                            })
                            ->visible(fn (Get $get) => $get('payable_type') === 'App\Models\Expense' && $get('expensable_type') === 'App\Models\Conveyancing' && filled($get('expensable_id'))),

                        Select::make('payable_id')
                            ->label('Related Entity')
                            ->options(function (Get $get) {
                                $direction = $get('direction');
                                $type = $get('payable_type');
                                $clientId = $get('client_id');

                                if ($direction === 'outgoing') {
                                    if ($type === 'App\Models\Salary' && $clientId) {
                                        return Salary::where('employee_id', $clientId)->get()->mapWithKeys(function ($salary) {
                                            $label = $salary->employee_name ?? $salary->name ?? 'Salary #' . $salary->id;
                                            if (isset($salary->period)) {
                                                $label .= ' (' . $salary->period . ')';
                                            }
                                            return [$salary->id => $label];
                                        });
                                    } elseif ($type === 'App\Models\OfficeUtility') {
                                        return OfficeUtility::all()->mapWithKeys(function ($utility) {
                                            $label = $utility->type ?? $utility->name ?? 'Utility #' . $utility->id;
                                            if (isset($utility->period)) {
                                                $label .= ' (' . $utility->period . ')';
                                            }
                                            return [$utility->id => $label];
                                        });
                                    }
                                } elseif ($direction === 'incoming' && $clientId) {
                                    if ($type === 'App\Models\Project') {
                                        return Project::whereHas('projectClients', function ($q) use ($clientId) {
                                            $q->where('client_id', $clientId);
                                        })->get()->mapWithKeys(function ($project) {
                                            return [$project->id => $project->name ?? 'Project #' . $project->id];
                                        });
                                    } elseif ($type === 'App\Models\Conveyancing') {
                                        return Conveyancing::where('client_id', $clientId)->get()->mapWithKeys(function ($convey) {
                                            return [$convey->id => $convey->serial_number ?? 'Conveyancing #' . $convey->id];
                                        });
                                    } elseif ($type === 'App\Models\Litigation') {
                                        return Litigation::where('client_id', $clientId)->get()->mapWithKeys(function ($lit) {
                                            return [$lit->id => $lit->case_number ?? 'Litigation #' . $lit->id];
                                        });
                                    }
                                }
                                return [];
                            })
                            ->searchable()
                            ->preload()
                            ->live()
                            ->afterStateUpdated(function (Set $set, Get $get, $state) {
                                $type = $get('payable_type');
                                $direction = $get('direction');
                                $amount = null;
                                if ($direction === 'outgoing') {
                                    if ($type === 'App\Models\Salary' && $state) {
                                        $salary = Salary::find($state);
                                        if ($salary) {
                                            $amount = $salary->amount;
                                        }
                                    } elseif ($type === 'App\Models\OfficeUtility' && $state) {
                                        $utility = OfficeUtility::find($state);
                                        if ($utility) {
                                            $amount = $utility->amount;
                                        }
                                    }
                                }
                                $set('amount_paid', $amount);
                                $set('paid_by', null);
                                $set('type', null);
                                $set('account_id', null);
                            })
                            ->visible(fn (Get $get) => filled($get('payable_type')) && $get('payable_type') !== 'App\Models\Expense' && 
                                ($get('direction') === 'outgoing' || filled($get('client_id')))),

                        Select::make('paid_by')
                            ->label('Paid By')
                            ->options(function (Get $get) {
                                $type = $get('payable_type');
                                $payableId = $get('payable_id');
                                if ($type === 'App\Models\Conveyancing' && $payableId) {
                                    $conveyancing = Conveyancing::with(['buyer.user', 'seller.user'])->find($payableId);
                                    $options = [];
                                    if ($conveyancing?->buyer) {
                                        $options[$conveyancing->buyer_id] = ($conveyancing->buyer->user?->name ?? 'Buyer') . ' (Buyer)';
                                    }
                                    if ($conveyancing?->seller) {
                                        $options[$conveyancing->seller_id] = ($conveyancing->seller->user?->name ?? 'Seller') . ' (Seller)';
                                    }
                                    return $options;
                                }
                                return [];
                            })
                            ->required(fn (Get $get) => $get('payable_type') === 'App\Models\Conveyancing')
                            ->live()
                            ->afterStateUpdated(function (Set $set, Get $get, $state) {
                                if ($get('payable_type') === 'App\Models\Conveyancing' && $get('payable_id')) {
                                    $conveyancing = Conveyancing::find($get('payable_id'));
                                    if ($conveyancing) {
                                        if ($state == $conveyancing->seller_id) {
                                            $set('type', 'legal_fees');
                                        } else {
                                            $set('type', null);
                                        }
                                    }
                                }
                            })
                            ->visible(fn (Get $get) => $get('payable_type') === 'App\Models\Conveyancing' && filled($get('payable_id'))),

                        Select::make('type')
                            ->label('Deposit Type')
                            ->options([
                                'purchase_price' => 'Purchase Price',
                                'legal_fees' => 'Legal Fees',
                            ])
                            ->required(fn (Get $get) => $get('payable_type') === 'App\Models\Conveyancing' && $get('paid_by') == Conveyancing::find($get('payable_id'))?->buyer_id)
                            ->visible(function (Get $get) {
                                if ($get('direction') !== 'incoming' || $get('payable_type') !== 'App\Models\Conveyancing') {
                                    return false;
                                }
                                $conveyancing = Conveyancing::find($get('payable_id'));
                                return $conveyancing && $get('paid_by') == $conveyancing->buyer_id;
                            }),

                        Hidden::make('type')
                            ->default(function (Get $get) {
                                $direction = $get('direction');
                                $type = $get('payable_type');
                                $payableId = $get('payable_id');
                                $paidBy = $get('paid_by');

                                if ($direction === 'incoming') {
                                    if ($type === 'App\Models\Litigation') {
                                        return 'legal_fees';
                                    } elseif ($type === 'App\Models\Conveyancing' && $payableId && $paidBy) {
                                        $conveyancing = Conveyancing::find($payableId);
                                        if ($conveyancing && $paidBy == $conveyancing->seller_id) {
                                            return 'legal_fees';
                                        }
                                    }
                                }
                                return null;
                            })
                            ->visible(function (Get $get) {
                                $type = $get('payable_type');
                                $direction = $get('direction');
                                if ($direction !== 'incoming') return false;
                                if ($type === 'App\Models\Litigation') return true;
                                if ($type === 'App\Models\Conveyancing') {
                                    $conveyancing = Conveyancing::find($get('payable_id'));
                                    return $conveyancing && $get('paid_by') == $conveyancing->seller_id;
                                }
                                return false;
                            }),

                        TextInput::make('amount_paid')
                            ->label('Amount (KSh)')
                            ->numeric()
                            ->required()
                            ->prefix('KSh')
                            ->minValue(0)
                            ->step(0.01),

                        Select::make('payment_method')
                            ->label('Payment Method')
                            ->options([
                                'mpesa' => 'M-Pesa',
                                'bank' => 'Bank Transfer',
                                'cash' => 'Cash',
                            ])
                            ->required()
                            ->live()
                            ->afterStateUpdated(function (Set $set) {
                                $set('account_id', null);
                            }),

                        Select::make('account_id')
                            ->label('Account')
                            ->options(Account::all()->mapWithKeys(function ($account) {
                                return [$account->id => $account->name . ' (' . $account->number . ')'];
                            }))
                            ->searchable()
                            ->preload()
                            ->required(fn (Get $get) => $get('payment_method') === 'bank')
                            ->visible(fn (Get $get) => $get('payment_method') === 'bank'),

                        TextInput::make('transaction_id')
                            ->label('Transaction ID')
                            ->placeholder('e.g., TXN123 (optional)'),

                        Hidden::make('status')
                            ->default('completed'),
                    ])
                    ->columns(3)
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('direction')
                    ->label('Direction')
                    ->formatStateUsing(fn ($state) => ucfirst($state))
                    ->sortable(),
                TextColumn::make('payable_type')
                    ->label('Type')
                    ->formatStateUsing(fn ($state) => str_replace('App\\Models\\', '', $state))
                    ->sortable(),
                TextColumn::make('payable_id')
                    ->label('Related Entity')
                    ->formatStateUsing(function ($state, Payment $record) {
                        try {
                            $payable = $record->payable;
                            if (!$payable) return 'Record #' . $state;

                            switch ($record->payable_type) {
                                case 'App\Models\Salary':
                                    $name = $payable->employee_name ?? $payable->name ?? 'Salary';
                                    $period = $payable->period ?? 'No Period';
                                    return $name . ' (' . $period . ')';
                                case 'App\Models\OfficeUtility':
                                    $type = $payable->type ?? $payable->name ?? 'Utility';
                                    $period = $payable->period ?? 'No Period';
                                    return $type . ' (' . $period . ')';
                                case 'App\Models\Project':
                                    return $payable->name ?? 'Project #' . $state;
                                case 'App\Models\Conveyancing':
                                    return $payable->serial_number ?? 'Conveyancing #' . $state;
                                case 'App\Models\Litigation':
                                    return $payable->case_number ?? 'Litigation #' . $state;
                                default:
                                    return 'Record #' . $state;
                            }
                        } catch (\Exception $e) {
                            return 'Error loading entity';
                        }
                    })
                    ->sortable(),
                TextColumn::make('amount_paid')
                    ->label('Amount')
                    ->money('KES')
                    ->sortable(),
                TextColumn::make('payment_method')
                    ->sortable(),
                TextColumn::make('status')
                    ->sortable()
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'pending' => 'warning',
                        'completed' => 'success',
                        'failed' => 'danger',
                    }),
                TextColumn::make('client.user.name')
                    ->label('Client')
                    ->default('No Client')
                    ->sortable(),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('direction')
                    ->options([
                        'outgoing' => 'Payment',
                        'incoming' => 'Deposit',
                    ]),
                SelectFilter::make('payable_type')
                    ->label('Type')
                    ->options([
                        'App\Models\Salary' => 'Salary',
                        'App\Models\OfficeUtility' => 'Office Utility',
                        'App\Models\Project' => 'Project',
                        'App\Models\Conveyancing' => 'Conveyancing',
                        'App\Models\Litigation' => 'Litigation',
                    ]),
                SelectFilter::make('status')
                    ->options([
                        'pending' => 'Pending',
                        'completed' => 'Completed',
                        'failed' => 'Failed',
                    ]),
            ])
            ->actions([
                Action::make('downloadReceipt')
                    ->label('Download Receipt')
                    ->icon('heroicon-o-document-arrow-down')
                    ->color('success')
                    ->url(fn (Payment $record): string => route('receipts.download', $record))
                    ->openUrlInNewTab(),
                   
                Action::make('viewReceipt')
                    ->label('View Receipt')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->url(fn (Payment $record): string => route('receipts.view', $record))
                    ->openUrlInNewTab(),
                   
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\CreatePointOfSale::route('/'),
            'view' => Pages\ViewPayment::route('/{record}'),
            'edit' => Pages\EditPayment::route('/{record}/edit'),
        ];
    }
}