<?php

namespace App\Filament\Resources\LitigationResource\Pages;

use Filament\Actions\Action;
use App\Filament\Resources\LitigationResource;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\ViewEntry;
use Barryvdh\DomPDF\Facade\Pdf;

class ViewLitigation extends ViewRecord
{
    protected static string $resource = LitigationResource::class;

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([

            Section::make('Litigation Details')->schema([
                ViewEntry::make('litigationDetails')
                    ->view('infolists.litigation-components.litigation-details-table')
                    ->viewData([
                        'case_number' => $this->record->case_number,
                        'court' => $this->record->station,
                        'court_assigned_case_number' => $this->record->court_assigned_case_number,
                        'type' => $this->record->litigationCaseType->name,
                        'case_category' => $this->record->litigationCategory->name,
                        'description' => $this->record->description,
                        'documents' => $this->record->documents,
                        'amount_charged' => $this->record->amount_charged,
                    ]),
            ]),


            //Table for client details
            Section::make('Client Details')->schema([
                ViewEntry::make('client')
                    ->view('infolists.litigation-components.litigation-client-details-table')
                    ->viewData([
                        'client' => $this->record->client,
                    ]),
            ]),

            //Advocates section
            Section::make('Parties')->schema([
                ViewEntry::make('litigationAdvocates')
                    ->view('infolists.litigation-components.litigation-advocate-details-table')
                    ->viewData([
                        'parties' => $this->record->litigationAdvocates,
                    ]),
            ]),

            //Stages section
            Section::make('Litigation Activities')->schema([
                ViewEntry::make('litigationActivities')
                    ->view('infolists.litigation-components.litigation-activity-details-table')
                    ->viewData([
                        'activities' => $this->record->litigationActivities,
                    ]),
            ]),

            /*
             Section::make('Payments')->schema([
                 ViewEntry::make('litigationPayments')
                     ->view('infolists.litigation-components.litigation-payments-details-table')
                     ->viewData([
                         'payments' => $this->record->litigationPayments,
                     ]),
             ]),
             */

            Section::make('General Litigation Expenses')
                ->schema([
                    ViewEntry::make('expenses')
                        ->view('infolists.litigation-components.litigation-expense-details-table')
                        ->viewData([
                            'expenses' => $this->record->expenses, // ✅ use the shared belongsToMany
                        ]),
                ]),


        ]);
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('Export PDF')
                ->label('Export to PDF')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->action(function () {
                    $litigation = $this->record->load([
                        'client.user',
                        'litigationAdvocates',
                        'litigationPayments',
                    ]);
                    // Extract related data
                    $client = $litigation->client;
                    $parties = $litigation->litigationAdvocates;
                    $payments = $litigation->litigationPayments;
                    $expenses = $litigation->litigationExpenses; // ✅ Add this
        
                    $pdf = Pdf::loadView('pdf.litigation', [
                        'litigation' => $litigation,
                        'client' => $client,
                        'parties' => $parties,
                        'payments' => $payments,
                        'expenses' => $expenses, // ✅ Pass to view
                    ]);

                    return response()->streamDownload(function () use ($pdf) {
                        echo $pdf->stream();
                    }, 'litigation-' . $litigation->id . '.pdf');
                }),

            //summary
            Action::make('Export Financial Summary')
                ->label('Export Financial Summary')
                ->icon('heroicon-o-document-chart-bar')
                ->color('primary')
                ->action(function () {
                    $litigation = $this->record->load([
                        'client.user',
                        'litigationPayments',
                        'litigationExpenses.expense.expenseItem',
                    ]);

                    // Log the litigationExpenses for debugging
                    \Log::info('Litigation Expenses for Litigation ID ' . $litigation->id, $litigation->litigationExpenses->toArray());

                    $pdf = Pdf::loadView('pdf.litigation-summary', [
                        'litigation' => $litigation,
                        'client' => $litigation->client,
                        'payments' => $litigation->litigationPayments,
                        'expenses' => $litigation->litigationExpenses,
                    ]);

                    return response()->streamDownload(function () use ($pdf) {
                        echo $pdf->stream();
                    }, 'litigation-summary-' . $litigation->id . '.pdf');
                }),

            Action::make('edit')
                ->label('Edit')
                ->icon('heroicon-o-pencil')
                ->url(fn() => LitigationResource::getUrl('edit', ['record' => $this->record])),


        ];
    }
}
