<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LitigationResource\Pages;
use App\Models\Advocate;
use App\Models\Client;
use App\Models\Litigation;
use App\Models\LitigationCaseType;
use App\Models\LitigationCategory;
use App\Models\LitigationStage;
use App\Models\User;
use App\Models\LitigationDocument;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\{
    Grid,
    Tabs,
    Tabs\Tab,
    Section,
    TextInput,
    Textarea,
    DatePicker,
    Select,
    Toggle,
    FileUpload,
    Repeater
};
use Filament\Tables\Columns\TextColumn;

class LitigationResource extends Resource
{
    protected static ?string $model = Litigation::class;

    //protected static ?string $navigationGroup = 'Litigation';
    protected static ?int $navigationSort = 2;

    protected static ?string $navigationIcon = 'heroicon-o-scale';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Section::make()
                ->schema([
                    Tabs::make('Litigation Form')->tabs([
                        Tab::make('Main Info')->schema([
                            Select::make('client_id')
                                ->label('Client')
                                ->searchable()
                                ->preload(true) // Fixed syntax error here
                                ->options(
                                    Client::with('user')->get()
                                        ->mapWithKeys(
                                            fn($client) =>
                                            optional($client->user)->name
                                            ? [$client->id => $client->user->name]
                                            : []
                                        )
                                )
                                ->required()
                                ->createOptionForm([
                                    Grid::make(2)->schema([
                                        TextInput::make('user.name')->label('Name')->required(),
                                        TextInput::make('phone')->required(),
                                        TextInput::make('national_id')->label('National ID')->required(),
                                        Textarea::make('address')->rows(2)->required()->columnSpanFull(),
                                    ])
                                ])
                                ->createOptionUsing(function (array $data) {
                                    $user = User::create([
                                        'name' => $data['user']['name'],
                                    ]);

                                    $user->assignRole('client');

                                    $client = Client::create([
                                        'user_id' => $user->id,
                                        'phone' => $data['phone'],
                                        'national_id' => $data['national_id'],
                                        'address' => $data['address'],
                                    ]);

                                    return $client->getKey();
                                }),

                            TextInput::make('case_number')
                                ->label('Case Number')
                                ->readOnly()
                                ->disabled()
                                ->dehydrated(false)
                                ->visible(fn($record) => filled($record))
                                ->columnSpanFull(),

                            TextInput::make('court_assigned_case_number')
                                ->label('Court Assigned Case Number')
                                ->nullable()
                                ->maxLength(255),
                            //Litigation Case type    
                            Select::make('litigation_case_type_id')
                                ->label('Case Type')
                                ->relationship('litigationCaseType', 'name')
                                ->searchable()
                                ->preload()
                                ->required()
                                ->createOptionForm([
                                    Forms\Components\TextInput::make('name')
                                        ->label('Case Type Name')
                                        ->required()
                                        ->unique(LitigationCaseType::class, 'name'),
                                ])
                                ->createOptionUsing(fn(array $data): int => LitigationCaseType::create($data)->id),

                            // Case Category
                            Select::make('litigation_category_id')
                                ->label('Case Category')
                                ->relationship('litigationCategory', 'name')
                                ->searchable()
                                ->preload()
                                ->required()
                                ->createOptionForm([
                                    Forms\Components\TextInput::make('name')
                                        ->label('Category Name')
                                        ->required()
                                        ->unique(LitigationCategory::class, 'name'),
                                ])
                                ->createOptionUsing(fn(array $data): int => LitigationCategory::create($data)->id),

                            TextInput::make('station')
                                ->label('Court Station')
                                ->maxLength(255)
                                ->nullable(),

                            DatePicker::make('filing_date')
                                ->label('Filing Date'),

                            TextInput::make('amount_charged')
                                ->label('Amount Charged')
                                ->numeric()
                                ->required()
                                ->reactive(),

                             Textarea::make('description')
                                ->label('Description')
                                ->required()
                                ->maxLength(1000),
                                
                        ])->columns(2),

                        Tab::make('Parties')->schema([
                            Repeater::make('litigationAdvocates')
                                ->label('Litigation Parties')
                                ->relationship('litigationAdvocates')
                                ->schema([
                                    TextInput::make('party')
                                        ->label('Name of Party')
                                        ->required()
                                        ->maxLength(255),

                                    Select::make('party_type')
                                        ->label('Party Type')
                                        ->options([
                                            '1st Plaintiff' => '1st Plaintiff',
                                            '1st Respondent' => '1st Respondent',
                                            '2nd Respondent' => '2nd Respondent',
                                            '1st Defendant' => '1st Defendant',
                                            '2nd Defendant' => '2nd Defendant',
                                        ])
                                        ->required(),

                                    TextInput::make('email')
                                        ->label('Email')
                                        ->email()
                                        ->nullable()
                                        ->maxLength(255),

                                    TextInput::make('phone')
                                        ->label('Phone Number')
                                        ->tel()
                                        ->nullable()
                                        ->maxLength(20),

                                    Select::make('gender')
                                        ->label('Gender')
                                        ->options([
                                            'Male' => 'Male',
                                            'Female' => 'Female',
                                            'Other' => 'Other',
                                        ])
                                        ->nullable(),
                                ])
                                ->defaultItems(1)
                                ->collapsible()
                                ->columns(2),
                        ]),

                        Tab::make('Activities')->schema([
                            Repeater::make('litigationActivities')
                                ->label('Litigation Activities')
                                ->relationship('litigationActivities')
                                ->schema([
                                    TextInput::make('activity')
                                        ->label('Activity')
                                        ->required()
                                        ->maxLength(255),

                                    TextInput::make('court_room')
                                        ->label('Court Room')
                                        ->required()
                                        ->maxLength(100),

                                    DatePicker::make('activity_date')
                                        ->label('Activity Date')
                                        ->required(),

                                    Textarea::make('outcome')
                                        ->label('Outcome')
                                        ->required()
                                        ->maxLength(1000),
                                ])
                                ->collapsible()
                                ->defaultItems(1)
                                ->columns(2),
                        ]),

                        Tab::make('Documents')->schema([
                            Repeater::make('documents')
                                ->label('Litigation Documents')
                                ->relationship('documents')
                                ->schema([
                                    TextInput::make('name')
                                        ->required()
                                        ->maxLength(255),
                                    FileUpload::make('file_path')
                                        ->disk('public')
                                        ->directory('litigation-documents')
                                        ->required(),
                                ])
                                ->mutateRelationshipDataBeforeCreateUsing(function (array $data, $record) {
                                    return [
                                        'documentable_id' => $record->id,
                                        'documentable_type' => Litigation::class,
                                        'name' => $data['name'],
                                        'file_path' => $data['file_path'],
                                    ];
                                })
                                ->collapsible()
                                ->defaultItems(0)
                                ->columns(2),
                        ]),
                    ])
                ])
                ->columnSpanFull()
                ->maxWidth('7xl'),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('client.user.name')->label('Client')->searchable(),
                TextColumn::make('case_number')->searchable(),
                TextColumn::make('filing_date')->date(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->with(['litigationPayments', 'litigationExpenses'])->latest();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLitigations::route('/'),
            'create' => Pages\CreateLitigation::route('/create'),
            'view' => Pages\ViewLitigation::route('/{record}'),
            'edit' => Pages\EditLitigation::route('/{record}/edit'),
        ];
    }
}