<?php

namespace App\Filament\Resources\InvoiceResource\Pages;

use App\Filament\Resources\InvoiceResource;
use App\Mail\InvoiceMail;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\ViewEntry;
use Filament\Actions\Action;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Filament\Notifications\Notification;

class ViewInvoice extends ViewRecord
{
    protected static string $resource = InvoiceResource::class;

    protected function getHeaderActions(): array
    {
        return [
            \Filament\Actions\EditAction::make(),
            \Filament\Actions\DeleteAction::make(),
            //invoice download
            Action::make('downloadInvoice')
                ->label('Download Invoice')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('success')
                ->action(function () {
                    $this->record->loadMissing(['client.user']); // ensure relations are eager loaded
        
                    $pdf = Pdf::loadView('pdf.invoice', [
                        'invoice' => $this->record,
                    ]);

                    // Sanitize the invoice number for a safe filename
                    $safeInvoiceNumber = preg_replace('/[\/\\\\:*?"<>|]/', '-', $this->record->invoice_number);

                    return response()->streamDownload(
                        fn() => print ($pdf->stream()),
                        "invoice-{$safeInvoiceNumber}.pdf"
                    );
                })
            ,
            //send invoice email
            Action::make('sendMail')
                ->label('Send Invoice Email')
                ->icon('heroicon-m-paper-airplane')
                //->color('success')
                ->requiresConfirmation()
                ->action(function ($record) {
                    // Adjust email access depending on your relationships
                    $email = $record->client?->user?->email;

                    if (!$email) {
                        Notification::make()
                            ->title('No email address found for the client.')
                            ->danger()
                            ->send();

                        return;
                    }

                    Mail::to($email)->send(new InvoiceMail($record));

                    Notification::make()
                        ->title('Invoice sent successfully to ' . $email)
                        ->success()
                        ->send();
                }),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            Section::make('Invoice Details')
                ->schema([
                    ViewEntry::make('invoice')
                        ->view('infolists.invoice-components.invoice-details')
                        ->viewData([
                            'invoice' => $this->record,
                        ]),
                ]),
        ]);
    }
}
