<?php

namespace App\Filament\Resources;

use App\Filament\Resources\InvoiceResource\Pages;
use App\Models\Invoice;
use App\Models\Client;
use App\Models\Litigation;
use App\Models\Conveyancing;
use App\Models\Project;
use App\Models\ProjectClient;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class InvoiceResource extends Resource
{
    protected static ?string $model = Invoice::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Financial Management';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Section::make('Invoice Details')->schema([

                // Step 1: Payment Type
                Select::make('payable_type')
                    ->label('Service Type')
                    ->options([
                        Project::class => 'Project',
                        Conveyancing::class => 'Conveyancing',
                        Litigation::class => 'Litigation',
                    ])
                    ->reactive()
                    ->required()
                    ->afterStateUpdated(fn(callable $set) => [
                        $set('payable_id', null),
                        $set('client_id', null),
                        $set('total_amount', null),
                    ]),

                // Step 2: Project / Related Entity
                Select::make('payable_id')
                    ->label(fn (callable $get) => $get('payable_type') === Project::class ? 'Project' : 'Related Entity')
                    ->options(function (callable $get) {
                        $type = $get('payable_type');

                        return match ($type) {
                            Project::class => Project::pluck('name', 'id')->toArray(),
                            Conveyancing::class => Conveyancing::pluck('serial_number', 'id')->toArray(),
                            Litigation::class => Litigation::pluck('case_number', 'id')->toArray(),
                            default => [],
                        };
                    })
                    ->searchable()
                    ->preload()
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $set, callable $get) {
                        $set('client_id', null);

                        // Auto-set client for Litigation
                        if ($get('payable_type') === Litigation::class && $state) {
                            $litigation = Litigation::with('client')->find($state);
                            if ($litigation && $litigation->client_id) {
                                $set('client_id', $litigation->client_id);
                            }
                        }
                    })
                    ->visible(fn(callable $get) => filled($get('payable_type'))),

                // Step 3: Client (Hidden for Litigation)
                Select::make('client_id')
                    ->label(fn (callable $get) => $get('payable_type') === Project::class ? 'Project Client' : 'Client')
                    ->options(function (callable $get) {
                        $type = $get('payable_type');
                        $payableId = $get('payable_id');

                        if ($type === Project::class && $payableId) {
                            return ProjectClient::where('project_id', $payableId)
                                ->with('client.user')
                                ->get()
                                ->mapWithKeys(fn($pc) => [$pc->id => $pc->client?->user?->name ?? 'N/A'])
                                ->toArray();
                        }

                        if ($type === Conveyancing::class && $payableId) {
                            $conv = Conveyancing::find($payableId);
                            if (!$conv) return [];

                            return [
                                $conv->buyer_id => 'Buyer - ' . ($conv->buyer?->user?->name ?? 'N/A'),
                                $conv->seller_id => 'Seller - ' . ($conv->seller?->user?->name ?? 'N/A'),
                            ];
                        }

                        return [];
                    })
                    ->searchable()
                    ->preload()
                    ->required(fn (callable $get) => $get('payable_type') !== Litigation::class)
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $set, callable $get) {
                        // Conveyancing auto-fill
                        if ($get('payable_type') === Conveyancing::class && $get('payable_id')) {
                            $conv = Conveyancing::find($get('payable_id'));
                            if ($conv) {
                                if ((int)$state === (int)$conv->buyer_id) {
                                    $set('total_amount', $conv->buyer_legal_fees);
                                } elseif ((int)$state === (int)$conv->seller_id) {
                                    $set('total_amount', $conv->seller_legal_fees);
                                }
                            }
                        }

                        // Project auto-fill from apartment or subdivision
                        if ($get('payable_type') === Project::class && $state) {
                            $pc = ProjectClient::with(['apartmentDetail', 'subdivisionDetail'])
                                ->find($state);

                            if ($pc) {
                                $amount = null;
                                if ($pc->apartmentDetail && $pc->apartmentDetail->amount_charged) {
                                    $amount = $pc->apartmentDetail->amount_charged;
                                } elseif ($pc->subdivisionDetail && $pc->subdivisionDetail->amount_charged) {
                                    $amount = $pc->subdivisionDetail->amount_charged;
                                }

                                if ($amount !== null) {
                                    $set('total_amount', $amount);
                                }
                            }
                        }
                    })
                    ->visible(fn(callable $get) =>
                        filled($get('payable_type')) &&
                        filled($get('payable_id')) &&
                        $get('payable_type') !== Litigation::class
                    ),

                // Hidden client_id for Litigation
                Hidden::make('client_id')
                    ->visible(fn (callable $get) => $get('payable_type') === Litigation::class),

                // Step 4: Amount
                TextInput::make('total_amount')
                    ->label('Invoice Amount Due')
                    ->prefix('KSh')
                    ->required()
                    ->numeric()
                    ->minValue(0),

                
                // Step 6: Notes
                TextInput::make('notes')
                    ->label('Notes')
                    ->nullable(),

                // Step 7: Invoice Number (view only on edit)
                TextInput::make('invoice_number')
                    ->label('Invoice Number')
                    ->disabled()
                    ->visible(fn(string $context) => $context !== 'create'),

            ])->columns(2)
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\TextColumn::make('invoice_number')->searchable(),
            Tables\Columns\TextColumn::make('client.user.name')->label('Client')->searchable(),
            Tables\Columns\TextColumn::make('payable_type')
                ->label('Type')
                ->formatStateUsing(fn(string $state) => class_basename($state))
                ->badge(),
            Tables\Columns\TextColumn::make('payable_display')
                ->label('Reference')
                ->getStateUsing(function ($record) {
                    return match ($record->payable_type) {
                        Litigation::class => optional($record->payable)->case_number ?? '—',
                        Conveyancing::class => optional($record->payable)->serial_number ?? '—',
                        Project::class => optional($record->payable)->name ?? '—',
                        default => '—',
                    };
                })
                ->searchable(),
           
            Tables\Columns\TextColumn::make('total_amount')->money('KES'),
            Tables\Columns\TextColumn::make('status')->badge(),
        ])
        ->filters([
            Tables\Filters\SelectFilter::make('status')->options([
                'unpaid' => 'Unpaid',
                'paid' => 'Paid',
                'overdue' => 'Overdue',
            ]),
        ])
        // ->defaultSort('issue_date', 'desc')
        ->actions([
            Tables\Actions\ViewAction::make(),
            Tables\Actions\EditAction::make(),
        ])
        ->bulkActions([
            Tables\Actions\BulkActionGroup::make([
                Tables\Actions\DeleteBulkAction::make(),
            ]),
        ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->latest();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListInvoices::route('/'),
            'create' => Pages\CreateInvoice::route('/create'),
            'view' => Pages\ViewInvoice::route('/{record}'),
            'edit' => Pages\EditInvoice::route('/{record}/edit'),
        ];
    }
}
