<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ExpenseResource\Pages;
use App\Models\Conveyancing;
use App\Models\Expense;
use App\Models\ExpenseItem;
use App\Models\Litigation;
use App\Models\ConveyancingExpense;
use App\Models\LitigationExpense;
use App\Models\Project;
use App\Models\ProjectExpense;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class ExpenseResource extends Resource
{
    protected static ?string $model = Expense::class;
    
     // Change the navigation label
    protected static ?string $navigationLabel = 'Funds Released';

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static ?string $navigationGroup = 'Financial Management';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Expense Details')
                    ->schema([
                        Select::make('expense_item_id')
                            ->label('Expense Item')
                            ->options(ExpenseItem::pluck('expense_item', 'id'))
                            ->required()
                            ->searchable()
                            ->createOptionForm([
                                TextInput::make('expense_item')
                                    ->label('Expense Item Name')
                                    ->required()
                                    ->maxLength(255)
                                    ->unique(ExpenseItem::class, 'expense_item'),
                            ])
                            ->createOptionUsing(function (array $data): int {
                                return ExpenseItem::create($data)->id;
                            }),
                        TextInput::make('amount_paid')
                            ->label('Amount Paid')
                            ->numeric()
                            ->required()
                            ->prefix('KES')
                            ->step(0.01),
                        TextInput::make('balance')
                            ->label('Balance')
                            ->numeric()
                            ->default(0.00)
                            ->prefix('KES')
                            ->step(0.01),
                        Select::make('payment_method')
                            ->label('Payment Method')
                            ->options([
                                'mpesa' => 'M-Pesa',
                                'bank' => 'Bank Transfer',
                                'cash' => 'Cash',
                            ])
                            ->required(),
                        Select::make('status')
                            ->label('Status')
                            ->options([
                                'pending' => 'Pending',
                                'completed' => 'Completed',
                                'failed' => 'Failed',
                            ])
                            ->required(),
                        Select::make('expensable_type')
                            ->label('Related To')
                            ->options([
                                Litigation::class => 'Litigation',
                                Conveyancing::class => 'Conveyancing',
                                Project::class => 'Project',
                            ])
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(function ($state, callable $set) {
                                $set('expensable_id', null);
                            }),
                        Select::make('expensable_id')
                            ->label('Related Record')
                            ->options(function (callable $get) {
                                $type = $get('expensable_type');
                                if ($type === Litigation::class) {
                                    return Litigation::pluck('case_number', 'id');
                                } elseif ($type === Conveyancing::class) {
                                    return Conveyancing::pluck('serial_number', 'id');
                                } elseif ($type === Project::class) {
                                    return Project::pluck('name', 'id');
                                }
                                return [];
                            })
                            ->required()
                            ->searchable(),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('expenseItem.expense_item')
                    ->label('Expense Item')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('amount_paid')
                    ->label('Amount Paid')
                    ->money('KES')
                    ->sortable(),
                Tables\Columns\TextColumn::make('balance')
                    ->label('Balance')
                    ->money('KES')
                    ->sortable(),
                Tables\Columns\TextColumn::make('payment_method')
                    ->label('Payment Method')
                    ->formatStateUsing(fn ($state) => Str::title(str_replace('_', ' ', $state)))
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->label('Status')
                    ->sortable(),
                Tables\Columns\TextColumn::make('expensable_type')
                    ->label('Related To')
                    ->formatStateUsing(fn ($state) => class_basename($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('expensable_id')
                    ->label('Related Record')
                    ->formatStateUsing(function ($state, $record) {
                        if ($record->expensable_type === Litigation::class) {
                            return Litigation::find($state)?->case_number ?? 'N/A';
                        } elseif ($record->expensable_type === Conveyancing::class) {
                            return Conveyancing::find($state)?->serial_number ?? 'N/A';
                        } elseif ($record->expensable_type === Project::class) {
                            return Project::find($state)?->name ?? 'N/A';
                        }
                        return 'N/A';
                    })
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('expensable_type')
                    ->label('Related To')
                    ->options([
                        Litigation::class => 'Litigation',
                        Conveyancing::class => 'Conveyancing',
                        Project::class => 'Project',
                    ]),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }
      public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->latest(); // Or ->orderBy('created_at', 'desc')
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListExpenses::route('/'),
            'create' => Pages\CreateExpense::route('/create'),
            'edit' => Pages\EditExpense::route('/{record}/edit'),
            'view' => Pages\ViewExpense::route('/{record}'),
            
        ];
    }

    public static function create(array $attributes = [], bool $shouldReturnModel = false)
    {
        return \DB::transaction(function () use ($attributes, $shouldReturnModel) {
            $expense = Expense::create([
                'expense_item_id' => $attributes['expense_item_id'],
                'amount_paid' => $attributes['amount_paid'],
                'balance' => $attributes['balance'],
                'payment_method' => $attributes['payment_method'],
                'status' => $attributes['status'],
                'expensable_id' => $attributes['expensable_id'],
                'expensable_type' => $attributes['expensable_type'],
            ]);

            if ($attributes['expensable_type'] === Litigation::class) {
                LitigationExpense::create([
                    'litigation_id' => $attributes['expensable_id'],
                    'expense_id' => $expense->id,
                ]);
            } elseif ($attributes['expensable_type'] === Conveyancing::class) {
                ConveyancingExpense::create([
                    'conveyancing_id' => $attributes['expensable_id'],
                    'expense_id' => $expense->id,
                ]);
            } elseif ($attributes['expensable_type'] === Project::class) {
                ProjectExpense::create([
                    'project_id' => $attributes['expensable_id'],
                    'expense_id' => $expense->id,
                ]);
            }

            return $shouldReturnModel ? $expense : true;
        });
    }
}