<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DocumentResource\Pages;
use App\Models\Document;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class DocumentResource extends Resource
{
    protected static ?string $model = Document::class;
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $pluralLabel = 'All Documents';
    protected static ?string $slug = 'documents';
    protected static ?string $recordTitleAttribute = 'name';

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canEdit($record): bool
    {
        return false;
    }

    // public static function canDelete($record): bool
    // {
    //     return false;
    // }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // File Type Image
                Tables\Columns\ImageColumn::make('file_type_icon')
                    ->label('Type')
                    ->alignCenter()
                    ->url(fn($record) => asset('storage/' . $record->file_path))
                    ->openUrlInNewTab()
                    ->getStateUsing(fn() => '') // no <img>, we handle background
                    ->extraAttributes(function ($record) {
                        $ext = strtolower(pathinfo($record->file_path, PATHINFO_EXTENSION));

                        $backgroundUrl = match ($ext) {
                            'pdf' => asset('images/pdf-icon.png'),
                            'doc', 'docx' => asset('images/word-icon.png'),
                            'xls', 'xlsx' => asset('images/excel-icon.png'),
                            'jpg', 'jpeg', 'png', 'gif', 'webp' => asset('images/image-icon.png'),
                            default => asset('images/file-icon.png'),
                        };

                        return [
                            'style' => "
                            background-image: url('{$backgroundUrl}');
                            background-size: cover;
                            background-position: center;
                            width: 50px;
                            height: 50px;
                            display: inline-block;
                            border-radius: 5px;
                            "
                        ];
                    }),




                // Document Name
                Tables\Columns\TextColumn::make('name')
                    ->label('Document Name')
                    ->alignCenter()
                    ->searchable()
                    ->sortable(),

                // Model Type
                Tables\Columns\TextColumn::make('model_type')
                    ->label('Model')
                    ->alignCenter()
                    ->getStateUsing(function (Document $record) {
                        return class_basename($record->documentable_type);
                    })
                    ->sortable()
                    ->searchable(query: function ($query, string $search) {
                        return $query->where('documentable_type', 'like', "%{$search}%");
                    }),


                // Associated Entity (dynamic)
                Tables\Columns\TextColumn::make('associated_entity')
                    ->label('Associated Entity')
                    ->alignCenter()
                    ->getStateUsing(function (Document $record) {
                        $type = class_basename($record->documentable_type);

                        return match ($type) {
                            'Project' => $record->documentable?->name
                            ? "Project: {$record->documentable->name}" : 'N/A',

                            'ProjectClient' => $record->documentable?->client?->user?->name
                            ? "Client: {$record->documentable->client->user->name}" : 'N/A',

                            'Conveyancing' => $record->documentable?->serial_number
                            ? "Conveyancing: {$record->documentable->serial_number}" : 'N/A',

                            'Litigation' => $record->documentable?->case_number
                            ? "Litigation: {$record->documentable->case_number}" : 'N/A',

                            default => 'N/A',
                        };
                    })
                    ->url(function (Document $record) {
                        $type = class_basename($record->documentable_type);

                        return match ($type) {
                            'Project' => route('filament.admin.resources.projects.view', $record->documentable_id),

                            'ProjectClient' => $record->documentable?->project
                            ? route('filament.admin.resources.projects.view', $record->documentable->project->id)
                            : null,

                            'Conveyancing' => route('filament.admin.resources.conveyancings.view', $record->documentable_id),

                            'Litigation' => route('filament.admin.resources.litigations.view', $record->documentable_id),

                            default => null,
                        };
                    })
                    ->openUrlInNewTab()
                    ->sortable()
                    ->searchable(query: function ($query, string $search) {
                        return $query->where(function ($q) use ($search) {
                            $q->whereHasMorph(
                                'documentable',
                                [
                                    \App\Models\Project::class,
                                    \App\Models\ProjectClient::class,
                                    \App\Models\Conveyancing::class,
                                    \App\Models\Litigation::class,
                                ],
                                function ($related) use ($search) {
                                    $model = $related->getModel();

                                    if ($model instanceof \App\Models\Project) {
                                        $related->where('name', 'like', "%{$search}%");
                                    }

                                    if ($model instanceof \App\Models\ProjectClient) {
                                        $related->whereHas('client.user', function ($clientQuery) use ($search) {
                                            $clientQuery->where('name', 'like', "%{$search}%");
                                        });
                                    }

                                    if ($model instanceof \App\Models\Conveyancing) {
                                        $related->where('serial_number', 'like', "%{$search}%");
                                    }

                                    if ($model instanceof \App\Models\Litigation) {
                                        $related->where('case_number', 'like', "%{$search}%");
                                    }
                                }
                            );
                        });
                    }),
                     Tables\Columns\TextColumn::make('serial_number')
                    ->label('Serial Number')
                    ->alignCenter()
                    ->searchable()
                    ->sortable(),


            ])
            ->filters([])
            ->actions([
                Tables\Actions\Action::make('download_document')
                    ->label('Download')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->color('primary')
                    ->url(function (Document $record) {
                        // Remove "storage/" prefix if present
                        $filePath = preg_replace('#^storage/#', '', $record->file_path);

                        if (!Storage::disk('public')->exists($filePath)) {
                            Log::error('File not found for download:', ['file_path' => $filePath]);
                            return null; // Hide button if missing
                        }

                        return asset('storage/' . $filePath);
                    })
                    ->openUrlInNewTab(),

                   //delete action
                Tables\Actions\DeleteAction::make(),


                    
            ])
            ->bulkActions([]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDocuments::route('/'),
        ];
    }
}
