<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DepositResource\Pages;
use App\Models\Conveyancing;
use App\Models\Litigation;
use App\Models\Payment;
use App\Models\Client;

use App\Models\Project;
use App\Models\ProjectClient;

use Filament\Forms\Components\Hidden;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;

class DepositResource extends Resource
{
    protected static ?string $model = Payment::class;
    // protected static ?string $navigationIcon = 'heroicon-o-arrow-down-circle';
    // protected static ?string $navigationLabel = 'Deposits';
    // protected static ?string $navigationGroup = 'Financial Management';
    protected static bool $shouldRegisterNavigation = false;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Deposit Details')
                    ->schema([
                        Select::make('payable_type')
                            ->label('Payment Type')
                            ->options([
                                'App\\Models\\Project' => 'Project',
                                'App\\Models\\Conveyancing' => 'Conveyancing',
                                'App\\Models\\Litigation' => 'Litigation',
                            ])
                            ->reactive()
                            ->required()
                            ->afterStateUpdated(fn(callable $set) => [
                                $set('payable_id', null),
                                $set('client_id', null),
                                $set('paid_by', null),
                                $set('type', null),
                            ]),

                        Select::make('client_id')
                            ->label('Client')
                            ->options(function (callable $get) {
                                $type = $get('payable_type');

                                return match ($type) {
                                    'App\\Models\\Litigation' => Client::whereHas('litigations')
                                        ->get()
                                        ->mapWithKeys(fn($client) => [$client->id => $client->user->name])
                                        ->toArray(),

                                    'App\\Models\\Conveyancing' => Client::whereHas('conveyancings')
                                        ->get()
                                        ->mapWithKeys(fn($client) => [$client->id => $client->user->name])
                                        ->toArray(),

                                    'App\\Models\\Project' => Client::whereHas('projectClients')
                                        ->get()
                                        ->mapWithKeys(fn($client) => [$client->id => $client->user->name])
                                        ->toArray(),

                                    default => [],
                                };
                            })
                            ->searchable()
                            ->preload()
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(fn(callable $set) => [
                                $set('payable_id', null),
                                $set('paid_by', null),
                                $set('type', null),
                            ])
                            ->visible(fn(callable $get) => filled($get('payable_type'))),

                        Select::make('payable_id')
                            ->label('Related Entity')
                            ->options(function (callable $get) {
                                $type = $get('payable_type');
                                $clientId = $get('client_id');

                                if (!$clientId)
                                    return [];

                                return match ($type) {
                                    'App\\Models\\Project' => Project::whereHas('projectClients', function ($q) use ($clientId) {
                                            $q->where('client_id', $clientId);
                                        })
                                        ->pluck('name', 'id')
                                        ->toArray(),

                                    'App\\Models\\Conveyancing' => Conveyancing::where('client_id', $clientId)
                                        ->pluck('serial_number', 'id')
                                        ->toArray(),

                                    'App\\Models\\Litigation' => Litigation::where('client_id', $clientId)
                                        ->pluck('case_number', 'id')
                                        ->toArray(),

                                    default => [],
                                };
                            })
                            ->searchable()
                            ->preload()
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(fn(callable $set, callable $get) => [
                                $set('paid_by', null),
                                $set('type', $get('payable_type') === 'App\\Models\\Litigation' ? 'legal_fees' : null),
                            ])
                            ->visible(fn(callable $get) => filled($get('payable_type')) && filled($get('client_id'))),

                        Select::make('paid_by')
                            ->label('Paid By')
                            ->options(function (callable $get) {
                                $conveyancingId = $get('payable_id');
                                if (!$conveyancingId || $get('payable_type') !== 'App\\Models\\Conveyancing') {
                                    return [];
                                }

                                $conveyancing = Conveyancing::with(['buyer.user', 'seller.user'])->find($conveyancingId);
                                if (!$conveyancing) {
                                    return [];
                                }

                                $options = [];
                                if ($conveyancing->buyer && $conveyancing->buyer->user) {
                                    $options[$conveyancing->buyer_id] = $conveyancing->buyer->user->name . ' (Buyer)';
                                }
                                if ($conveyancing->seller && $conveyancing->seller->user) {
                                    $options[$conveyancing->seller_id] = $conveyancing->seller->user->name . ' (Seller)';
                                }

                                return $options;
                            })
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(fn(callable $set, callable $get) => [
                                $set('type', $get('payable_type') === 'App\\Models\\Conveyancing' && $get('paid_by') == Conveyancing::find($get('payable_id'))?->seller_id ? 'legal_fees' : null),
                            ])
                            ->visible(
                                fn(callable $get) =>
                                $get('payable_type') === 'App\\Models\\Conveyancing' &&
                                filled($get('client_id')) &&
                                filled($get('payable_id'))
                            ),

                        Select::make('type')
                            ->label('Deposit Type')
                            ->options([
                                'purchase_price' => 'Purchase Price',
                                'legal_fees' => 'Legal Fees',
                            ])
                            ->required()
                            ->visible(
                                fn(callable $get) =>
                                $get('payable_type') === 'App\\Models\\Conveyancing' &&
                                filled($get('client_id')) &&
                                filled($get('payable_id')) &&
                                filled($get('paid_by')) &&
                                $get('paid_by') == Conveyancing::find($get('payable_id'))?->buyer_id
                            ),

                        Hidden::make('type')
                            ->default('legal_fees')
                            ->required()
                            ->visible(
                                fn(callable $get) =>
                                ($get('payable_type') === 'App\\Models\\Conveyancing' &&
                                    filled($get('client_id')) &&
                                    filled($get('payable_id')) &&
                                    filled($get('paid_by')) &&
                                    $get('paid_by') == Conveyancing::find($get('payable_id'))?->seller_id) ||
                                ($get('payable_type') === 'App\\Models\\Litigation' &&
                                    filled($get('client_id')) &&
                                    filled($get('payable_id')))
                            ),

                        TextInput::make('amount_paid')
                            ->numeric()
                            ->required()
                            ->prefix('KSh')
                            ->minValue(0),

                        Select::make('payment_method')
                            ->options([
                                'mpesa' => 'M-Pesa',
                                'bank' => 'Bank Transfer',
                                'cash' => 'Cash',
                            ])
                            ->required(),

                        TextInput::make('transaction_id')
                            ->nullable()
                            ->placeholder('e.g., TXN123'),

                    ])
                    ->columns(2),
            ]);
    }

     public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // Show client name
                TextColumn::make('client.user.name')
                    ->label('Client Name')
                    ->sortable()
                    ->alignCenter()
                    ->searchable(),

                // Show type of paymentable model (e.g. Project, Litigation)
                TextColumn::make('payable_type')
                    ->label('Type')
                    ->alignCenter()
                    ->formatStateUsing(fn(string $state) => class_basename($state))
                    ->badge(),

                // Related item display (project name, conveyancing ID, litigation case number)
                TextColumn::make('payable_id')
                    ->label('Related Entity')
                    ->alignCenter()
                    ->formatStateUsing(function ($state, Payment $record) {
                        if ($record->payable_type === 'App\Models\Project') {
                            return $record->payable?->name ?? 'Project #' . $state;
                        } elseif ($record->payable_type === 'App\Models\Conveyancing') {
                            return $record->payable?->serial_number ?? 'Conveyancing #' . $state;
                        } elseif ($record->payable_type === 'App\Models\Litigation') {
                            return $record->payable?->case_number ?? 'Litigation #' . $state;
                        }
                        return $state;
                    })
                    ->searchable(query: function ($query, $search) {
                        $query->where(function ($query) use ($search) {
                            $query->whereHasMorph(
                                'payable',
                                ['App\Models\Project'],
                                fn($q) => $q->where('name', 'like', "%{$search}%")
                            )->orWhereHasMorph(
                                    'payable',
                                    ['App\Models\Conveyancing'],
                                    fn($q) => $q->where('serial_number', 'like', "%{$search}%")
                                )->orWhereHasMorph(
                                    'payable',
                                    ['App\Models\Litigation'],
                                    fn($q) => $q->where('case_number', 'like', "%{$search}%")
                                );
                        });
                    })
                    ->url(function ($record) {
                        if ($record->payable_type === 'App\Models\Project') {
                            return \App\Filament\Resources\ProjectResource::getUrl('view', ['record' => $record->payable_id]);
                        }
                        return null;
                    })
                    ->openUrlInNewTab(),
                // Amount column formatted as KES
                TextColumn::make('amount_paid')
                    ->money('KES')
                    ->alignCenter()
                    ->sortable(),


                // Payment method column
                TextColumn::make('payment_method')
                    ->alignCenter()
                    ->sortable(),

                // Status column with color badges
                // TextColumn::make('status')
                //     ->sortable()
                //     ->alignCenter()
                //     ->badge()
                //     ->color(fn(string $state): string => match ($state) {
                //         'pending' => 'warning',
                //         'completed' => 'success',
                //         'failed' => 'danger',
                //     }),

                // Created at column (timestamp)
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('client_id')
                    ->label('Client')
                    ->options(function () {
                        return Client::whereHas('payments', function ($query) {
                            $query->whereHasMorph('payable', [
                                'App\Models\Litigation',
                                'App\Models\Conveyancing',
                                'App\Models\ProjectClient',
                            ]);
                        })
                            ->with('user')
                            ->get()
                            ->pluck('user.name', 'id');
                    })
                    ->searchable(),

            ])
            ->actions([
                // Default view/edit/delete actions
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),

            ])
            ->bulkActions([
                // Bulk delete action
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }


    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->with([
            'client.user',
            'payable' => function ($morphTo) {
                $morphTo->morphWith([
                    ProjectClient::class => ['project'],
                    Conveyancing::class => [],
                    Litigation::class => [],
                ]);
            },
        ])->latest();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDeposits::route('/'),
            'create' => Pages\CreateDeposit::route('/create'),
            'view' => Pages\ViewDeposit::route('/{record}'),
            'edit' => Pages\EditDeposit::route('/{record}/edit'),
        ];
    }
}
