<?php
namespace App\Filament\Resources\ConveyancingResource\Pages;

use App\Filament\Resources\ConveyancingResource;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\ViewEntry;
use Filament\Actions\Action;
use Barryvdh\DomPDF\Facade\Pdf;

class ViewConveyancing extends ViewRecord
{
    protected static string $resource = ConveyancingResource::class;

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([

            Section::make('Conveyancing Details')->schema([
                ViewEntry::make('conveyancingDetails')
                    ->view('infolists.conveyancing-components.conveyancing-details-table')
                    ->viewData([
                        'serial_number' => $this->record->serial_number,
                        'property_address' => $this->record->property_address,
                        'sale_price' => $this->record->sale_price,
                        'status' => $this->record->status,
                        'contract_date' => $this->record->contract_date,
                        'settlement_date' => $this->record->completion_period,
                        'notes' => $this->record->notes,
                        'documents' => $this->record->documents, // ← Add this line
                    ]),
            ]),

            Section::make('Client Details')->schema([
                ViewEntry::make('client')
                    ->view('infolists.conveyancing-components.conveyancing-client-details-table')
                    ->viewData([
                        'client' => $this->record->client,
                    ]),
            ]),

            Section::make('Parties Involved')->schema([
                ViewEntry::make('parties')
                    ->view('infolists.conveyancing-components.conveyancing-parties-details-table')
                    ->viewData([
                        'buyer_name' => optional($this->record->buyer?->user)->name ?? 'N/A',
                        'seller_name' => optional($this->record->seller?->user)->name ?? 'N/A',
                    ]),
            ]),

            Section::make('Advocates')->schema([
                ViewEntry::make('advocates')
                    ->view('infolists.conveyancing-components.conveyancing-advocate-details-table')
                    ->viewData([
                        'advocates' => $this->record->conveyancingAdvocates,
                    ]),
            ]),


            Section::make('Payments')->schema([
                ViewEntry::make('payments')
                    ->view('infolists.conveyancing-components.conveyancing-payments-details-table')
                    ->viewData([
                        'payments' => $this->record->conveyancingPayments,
                        'record' => $this->record, // 👈 add this line
                    ]),
            ]),


            Section::make('Expenses')->schema([
                ViewEntry::make('expenses')
                    ->view('infolists.conveyancing-components.conveyancing-expense-details-table')
                    ->viewData([
                        'expenses' => $this->record->expenses, // use morphMany
                    ]),
            ])
        ]);
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('Export PDF')
                ->label('Export to PDF')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->visible(fn() => auth()->user()->hasAnyRole(['admin', 'super_admin']))
                ->action(function () {
                    $record = $this->record->load([
                        'client.user',
                        'buyer.user',
                        'seller.user',
                        'conveyancingAdvocates.advocate.user',
                        'conveyancingPayments',
                        'expenses',
                    ]);

                    $pdf = Pdf::loadView('pdf.conveyancing', [
                        'record' => $record,
                        'client' => $record->client,
                        'advocates' => $record->conveyancingAdvocates,
                        'payments' => $record->conveyancingPayments,
                        'expenses' => $record->expenses,
                    ]);

                    return response()->streamDownload(
                        fn() => print ($pdf->stream()),
                        'conveyancing-details-' . $record->id . '.pdf'
                    );
                }),


            //now for summary 
            Action::make('Export Summary PDF')
                ->label('Export Financial Summary')
                ->icon('heroicon-o-document-chart-bar')
                ->color('primary')
                ->visible(fn() => auth()->user()->hasAnyRole(['admin', 'super_admin']))
                ->action(function () {
                    $conveyancing = $this->record->load([
                        'conveyancingPayments',
                        'expenses',
                    ]);

                    $payments = $conveyancing->conveyancingPayments;
                    $expenses = $conveyancing->expenses; 
                    $pdf = Pdf::loadView('pdf.conveyancing-summary', [
                        'record' => $conveyancing,
                        'payments' => $payments,
                        'expenses' => $expenses,
                    ]);

                    return response()->streamDownload(function () use ($pdf) {
                        echo $pdf->stream();
                    }, 'conveyancing-summary-' . $conveyancing->id . '.pdf');
                }),

            Action::make('edit')
                ->label('Edit')
                ->icon('heroicon-o-pencil')
                ->url(fn() => ConveyancingResource::getUrl('edit', ['record' => $this->record])),

        ];
    }
}
