<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ConveyancingResource\Pages;
use App\Models\Advocate;
use App\Models\Client;
use App\Models\Conveyancing;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\Grid;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Tabs\Tab;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class ConveyancingResource extends Resource
{
    protected static ?string $model = Conveyancing::class;

    // protected static ?string $navigationGroup = 'Conveyancing';

    protected static ?int $navigationSort = 3;

    protected static ?string $navigationIcon = 'heroicon-o-home-modern';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Section::make()
                ->schema([
                    Tabs::make('Conveyancing Form')->tabs([
                        Tab::make('Main Info')->schema([
                            Select::make('client_id')
                                ->label('Client')
                                ->searchable()
                                ->preload()
                                ->options(
                                    Client::with('user')->get()
                                        ->mapWithKeys(
                                            fn($client) =>
                                            optional($client->user)->name
                                            ? [$client->id => $client->user->name]
                                            : []
                                        )
                                )
                                ->required()
                                ->createOptionForm([
                                    Grid::make(2)->schema([
                                        TextInput::make('user.name')->label('Name')->required(),
                                       TextInput::make('phone')->label('Phone')->unique(
                                            table: 'clients',
                                            column: 'phone',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('national_id')->label('National ID')
                                        ->unique(
                                            table: 'clients',
                                            column: 'national_id',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('address')->label('Address')->required(),
                                    ])
                                ])
                                ->createOptionUsing(function (array $data) {
                                    $user = User::create([
                                        'name' => $data['user']['name'],
                                    ]);
                                    $user->assignRole('client');

                                    $client = Client::create([
                                        'user_id' => $user->id,
                                        'phone' => $data['phone'],
                                        'national_id' => $data['national_id'],
                                        'address' => $data['address'],
                                    ]);

                                    return $client->getKey();
                                }),

                            TextInput::make('serial_number')
                                ->label('Serial Number')
                                ->readOnly()
                                ->disabled()
                                ->dehydrated(false)
                                ->visible(fn($record) => filled($record))
                                ->columnSpanFull(),

                            TextInput::make('property_address')->required()->maxLength(255),

                            Select::make('seller_id')
                                ->label('Seller')
                                ->searchable()
                                ->preload()
                                ->options(
                                    Client::with('user')->get()
                                        ->mapWithKeys(fn($client) => [
                                            $client->id => optional($client->user)->name
                                        ])
                                        ->filter()
                                )
                                ->createOptionForm([
                                    Grid::make(2)->schema([
                                        TextInput::make('user.name')->label('Name')->required(),
                                        TextInput::make('phone')->label('Phone')->unique(
                                            table: 'clients',
                                            column: 'phone',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('national_id')->label('National ID')
                                        ->unique(
                                            table: 'clients',
                                            column: 'national_id',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('address')->label('Address')->required(),
                                    ])
                                ])
                                ->createOptionUsing(function (array $data) {
                                    $user = User::create([
                                        'name' => $data['user']['name'],
                                    ]);
                                    $user->assignRole('client');

                                    $client = Client::create([
                                        'user_id' => $user->id,
                                        'phone' => $data['phone'],
                                        'national_id' => $data['national_id'],
                                        'address' => $data['address'],
                                    ]);

                                    return $client->getKey();
                                })
                                ->required(),

                            Select::make('buyer_id')
                                ->label('Buyer')
                                ->searchable()
                                ->preload()
                                ->options(
                                    Client::with('user')->get()
                                        ->mapWithKeys(fn($client) => [
                                            $client->id => optional($client->user)->name
                                        ])
                                        ->filter()
                                )
                                ->createOptionForm([
                                    Grid::make(2)->schema([
                                        TextInput::make('user.name')->label('Name')->required(),
                                        TextInput::make('phone')->label('Phone')->unique(
                                            table: 'clients',
                                            column: 'phone',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('national_id')->label('National ID')
                                        ->unique(
                                            table: 'clients',
                                            column: 'national_id',
                                            ignorable: fn($record) => $record
                                        )->required(),
                                        TextInput::make('address')->label('Address')->required(),
                                    ])
                                ])
                                ->createOptionUsing(function (array $data) {
                                    $user = User::create([
                                        'name' => $data['user']['name'],
                                    ]);
                                    $user->assignRole('client');

                                    $client = Client::create([
                                        'user_id' => $user->id,
                                        'phone' => $data['phone'],
                                        'national_id' => $data['national_id'],
                                        'address' => $data['address'],
                                    ]);

                                    return $client->getKey();
                                })
                                ->required(),

                            TextInput::make('sale_price')->numeric()->required()->prefix('Ksh'),
                            TextInput::make('buyer_legal_fees')
                                ->label('Buyer Legal Fees')
                                ->numeric()
                                ->required()
                                ->prefix('Ksh'),
                            TextInput::make('seller_legal_fees')
                                ->label('Seller Legal Fees')
                                ->numeric()
                                ->required()
                                ->prefix('Ksh'),



                            DatePicker::make('contract_date'),
                            DatePicker::make('completion_period'),
                            Textarea::make('notes')->maxLength(1000),
                        ])->columns(2),

                        Tab::make('Advocates')->schema([
                            Repeater::make('conveyancingAdvocates')
                                ->relationship('conveyancingAdvocates')
                                ->schema([
                                    Select::make('advocate_id')
                                        ->label('Advocate')
                                        ->searchable()
                                        ->preload()
                                        ->options(
                                            Advocate::with('user')->get()
                                                ->mapWithKeys(fn($advocate) => [
                                                    $advocate->id => $advocate->user->name ?? 'Unknown',
                                                ])
                                                ->toArray()
                                        )
                                        ->disableOptionWhen(function ($value, callable $get, callable $livewire) {
                                            $all = $get('../../conveyancingAdvocates') ?? [];
                                            $selectedIds = collect($all)->pluck('advocate_id')->filter()->toArray();
                                            $currentIndex = $get('__index');
                                            unset($selectedIds[$currentIndex]);
                                            return in_array($value, $selectedIds);
                                        })
                                        ->required(),

                                    Select::make('role')
                                        ->label('Role')
                                        ->options([
                                            'Lead' => 'Lead',
                                            'Assistant' => 'Assistant',
                                        ])
                                        ->default(fn(Forms\Get $get) => $get('__repeater_index') == 0 ? 'Lead' : 'Assistant')
                                        ->required(),
                                ])
                                ->defaultItems(1)
                                ->collapsible()
                                ->columns(2),
                        ]),

                        Tab::make('Documents')->schema([
                            Repeater::make('documents')
                                ->label('Conveyancing Documents')
                                ->relationship('documents')
                                ->schema([
                                    TextInput::make('name')
                                        ->required()
                                        ->maxLength(255),
                                    FileUpload::make('file_path')
                                        ->disk('public')
                                        ->directory('conveyancing-documents')
                                        ->required(),
                                ])
                                ->mutateRelationshipDataBeforeCreateUsing(function (array $data, $record) {
                                    return [
                                        'documentable_id' => $record->id,
                                        'documentable_type' => Conveyancing::class,
                                        'name' => $data['name'],
                                        'file_path' => $data['file_path'],
                                    ];
                                })
                                ->collapsible()
                                ->defaultItems(0)
                                ->columns(2),
                        ]),
                    ])
                ])
                ->columnSpanFull()
                ->maxWidth('7xl'),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('client.user.name')->label('Client')->searchable(),
                TextColumn::make('serial_number')->label('Serial Number')->searchable()->sortable(),
                TextColumn::make('property_address')->searchable(),
                TextColumn::make('seller.user.name')->label('Seller')->sortable()->searchable(),
                TextColumn::make('buyer.user.name')->label('Buyer')->sortable()->searchable(),
                TextColumn::make('sale_price')->label('Price')->money('KES'),
                // TextColumn::make('status')->badge()->color(fn(string $state) => match ($state) {
                //     'pending' => 'gray',
                //     'in_progress' => 'info',
                //     'completed' => 'success',
                //     'cancelled' => 'danger',
                //     default => 'secondary',
                // }),
                TextColumn::make('contract_date')->date(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->latest();
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListConveyancings::route('/'),
            'create' => Pages\CreateConveyancing::route('/create'),
            'view' => Pages\ViewConveyancing::route('/{record}'),
            'edit' => Pages\EditConveyancing::route('/{record}/edit'),
        ];
    }
}