<?php

namespace App\Filament\Resources\ClientResource\Pages;

use App\Filament\Resources\ClientResource;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\ViewEntry;

use Filament\Actions\Action;
use Barryvdh\DomPDF\Facade\Pdf;

class ViewClient extends ViewRecord
{
    protected static string $resource = ClientResource::class;
    
    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            Section::make('Client Information')->schema([
                ViewEntry::make('client')
                    ->view('infolists.client-components.client-details-table')
                    ->viewData([
                        'client' => $this->record,
                    ]),
            ]),
            Section::make('Current Litigations')->schema([
                ViewEntry::make('litigations')
                    ->view('infolists.client-components.client-litigation-cases-table')
                    ->viewData([
                        'litigations' => $this->record->litigations,
                    ]),
            ]),
            Section::make('Litigation Payment Information')->schema([
                ViewEntry::make('litigation_payments')
                    ->view('infolists.client-components.client-litigation-payments-table')
                    ->viewData([
                        'payments' => $this->record->litigationPayments,
                        'total' => $this->record->litigationPayments->sum('amount_paid'),
                    ]),
            ]),
            Section::make('Current Conveyancing Information')->schema([
                ViewEntry::make('conveyancings')
                    ->view('infolists.client-components.client-conveyancing-table')
                    ->viewData([
                        'conveyancings' => $this->record->conveyancings,
                    ]),
            ]),
            Section::make('Conveyancing Payments Received')->schema([
                ViewEntry::make('conveyancing_payments')
                    ->view('infolists.client-components.client-conveyancing-payments-table')
                    ->viewData([
                        'payments' => $this->record->conveyancingPayments,
                    ]),
            ]),

            // ✅ New Section for Projects
            Section::make('Projects Information')->schema([
                ViewEntry::make('projects')
                    ->view('infolists.client-components.client-projects-table')
                    ->viewData([
                        'projectClients' => $this->record->projectClients()->with(['project', 'apartmentDetail', 'subdivisionDetail'])->get(),
                    ]),
            ]),

            //Section for client documents
            Section::make('Client Documents')
                ->schema([
                    ViewEntry::make('allDocuments')
                        ->view('infolists.client-components.client-documents-table')
                        ->viewData([
                            'clientDocuments' => $this->record->allDocuments(),
                        ]),
                ]),

            


        ]);
    }



    protected function getHeaderActions(): array
    {
        return [
            Action::make('Client Financial Report')
                ->label('Client report')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->action(function () {
                    $record = $this->record;

                    // Load conveyancing payments
                    $conveyancingPayments = \App\Models\ConveyancingPayment::where('client_id', $record->id)
                        ->orWhere('paid_by', $record->id)
                        ->with(['conveyancing.client.user', 'client.user', 'paidBy.user'])
                        ->get();

                    // Load client relationships
                    $record->load([
                        'litigationPayments.litigation',
                        'projectClientPayments.projectClient.project', // Use new relationship
                        'projectClientPayments.projectClient.client.user',
                        'projectClients.client.user',
                    ]);

                    $pdf = Pdf::loadView('pdf.client', [
                        'client' => $record,
                        'conveyancingPayments' => $conveyancingPayments,
                        'litigationPayments' => $record->litigationPayments,
                        'projectClientPayments' => $record->projectClientPayments, // Use new relationship
                        'projectClients' => $record->projectClients,
                    ]);

                    return response()->streamDownload(
                        fn() => print ($pdf->output()),
                        'client-financial-report-' . $record->id . '.pdf'
                    );
                }),
            Action::make('edit')
                ->label('Edit')
                ->icon('heroicon-o-pencil')
                ->url(fn() => ClientResource::getUrl('edit', ['record' => $this->record])),

        ];
    }

}
