<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ClientResource\Pages;
use App\Models\Client;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Hash;

class ClientResource extends Resource
{
    // The Eloquent model this resource manages
    protected static ?string $model = Client::class;

    // The sidebar group where this resource will appear
    protected static ?string $navigationGroup = 'Admin';

    // Icon for this resource in the sidebar
    protected static ?string $navigationIcon = 'heroicon-o-user';

    /**
     * Define the form schema for creating or editing a client.
     */
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // Section for the user's personal details (from users table)
                Section::make('User Information')
                    ->schema([
                        // Client's name (from related user)
                        TextInput::make('name')
                            ->required()
                            ->maxLength(255),

                        // Email (unique in users table, required)
                        TextInput::make('email')
                            ->email()
                            ->required()
                            ->unique(
                                table: 'users',
                                column: 'email',
                                ignorable: fn($record) => $record?->user
                            ),

                        // Password (only required when creating, hashed before saving)
                        // TextInput::make('password')
                        //     ->password()
                        //     ->required()
                        //     ->minLength(8)
                        //     ->dehydrateStateUsing(fn($state) => Hash::make($state))
                        //     ->visibleOn('create'),
                    ])
                    ->columns(2), // Layout: two-column

                // Section for client-specific information
                Section::make('Client Information')
                    ->schema([
                        // Phone number (must be unique in clients table)
                        TextInput::make('phone')
                           ->label('Phone number')
                            ->required()
                            ->unique(
                                table: 'clients',
                                column: 'phone',
                                ignorable: fn($record) => $record
                            ),

                        // National ID (must be unique in clients table)
                        TextInput::make('national_id')
                            ->label('National ID')
                            ->maxLength(11)
                            ->required()
                            ->unique(
                                table: 'clients',
                                column: 'national_id',
                                ignorable: fn($record) => $record
                            ),

                        // Address (optional, max 255 characters)
                        TextInput::make('address')
                           ->label('Address')
                            ->maxLength(255)
                            ->nullable(),

                        //KRA Pin    
                        TextInput::make('kra_pin')
                            ->label('Kra Pin')
                            ->maxLength(255)
                            ->nullable(), 
                        
                        //Passport Number
                        TextInput::make('passport_number')
                            ->label('Passport Number')
                            ->maxLength(255)
                            ->nullable(),     
                        //Nationality
                         TextInput::make('nationality')
                            ->label('Nationality')
                            ->maxLength(255)
                            ->nullable(),
                          
                        
                        //identification image
                        FileUpload::make('identification_image')
                                ->disk('public')
                                ->directory('client-identifications')
                                ->maxSize(2048)
                                ->nullable(),
    
                        
                            
                    ])->columns(2), // Layout: two-column
            ]);
    }

    /**
     * Define the table view shown in the resource's index page.
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // Display client name (from related user)
                Tables\Columns\TextColumn::make('user.name')
                    ->sortable()
                    ->searchable(),

                // Phone number
                Tables\Columns\TextColumn::make('phone')
                    ->sortable(),

                // National ID
                Tables\Columns\TextColumn::make('national_id')
                    ->sortable(),

                // Address
                Tables\Columns\TextColumn::make('address')
                    ->sortable(),

                // Date the client record was created
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime(),
            ])
            ->filters([
                // Add filters here if needed
            ])
            ->actions([
                // View individual client
                Tables\Actions\ViewAction::make(),

                // Edit client details
                Tables\Actions\EditAction::make(),

                // Delete client
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                // Bulk delete multiple clients
                Tables\Actions\DeleteBulkAction::make(),
            ])
            // Clicking on a row will navigate to the view page
            ->recordUrl(fn($record) => ClientResource::getUrl('view', ['record' => $record]))

            // Sort by newest first
            ->defaultSort('created_at', 'desc');
    }

    /**
     * Define the routes/pages for this resource.
     */
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListClients::route('/'),
            'create' => Pages\CreateClient::route('/create'),
            'view' => Pages\ViewClient::route('/{record}'),
            'edit' => Pages\EditClient::route('/{record}/edit'),
        ];
    }

    /**
     * Customize the Eloquent query to eager-load relationships.
     */
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->with([
                'user', // Load associated user
                'litigations', // Load related litigations (cases)
                'litigationPayments.litigation', // Load payments + their related litigations
            ]);
    }
}
