<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AdvocateResource\Pages;
use App\Models\Advocate;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Hash;

class AdvocateResource extends Resource
{
    // Set the model this resource manages
    protected static ?string $model = Advocate::class;

    // Group in Filament sidebar
    protected static ?string $navigationGroup = 'Admin';

    // Icon for this resource in the sidebar
    protected static ?string $navigationIcon = 'heroicon-o-user';

    /**
     * Define the form used for creating/editing an Advocate.
     */
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // Section for basic user info
                Forms\Components\Section::make('User Information')
                    ->schema([
                        // Name input (required)
                        TextInput::make('name')
                            ->required()
                            ->maxLength(255),

                        // Email input (required, must be unique across users table)
                        TextInput::make('email')
                            ->required()
                            ->email()
                            ->unique(
                                table: 'users',
                                column: 'email',
                                ignorable: fn($record) => $record?->user
                            ),

                        // Password input (only shown when creating, hashed before saving)
                        TextInput::make('password')
                            ->password()
                            ->required()
                            ->minLength(8)
                            ->dehydrateStateUsing(fn($state) => Hash::make($state))
                            ->visibleOn('create'),
                    ])
                    ->columns(2), // Arrange above fields in 2 columns

                // Section for advocate-specific info
                Section::make('Advocate Information')
                    ->schema([
                        // License number (nullable, must be unique if provided)
                        TextInput::make('license_number')
                            ->maxLength(255)
                            ->nullable()
                            ->unique(
                                table: 'users',
                                column: 'email', // (This may be a bug: should probably be 'license_number')
                                ignorable: fn($record) => $record?->user
                            ),

                        // Dropdown for selecting specialization
                        Select::make('specialization')
                            ->label('Specialization')
                            ->options([
                                'Property Law' => 'Property Law',
                                'Conveyancing' => 'Conveyancing',
                                'Real Estate & Development Law' => 'Real Estate & Development Law',
                                'Land Use & Zoning Law' => 'Land Use & Zoning Law',
                                'Landlord-Tenant (Tenancy) Law' => 'Landlord-Tenant (Tenancy) Law',
                                'Environmental & Land Use Law' => 'Environmental & Land Use Law',
                                'Construction Law' => 'Construction Law',
                                'Succession Law (Property Inheritance)' => 'Succession Law (Property Inheritance)',
                                'Trusts & Estate Law' => 'Trusts & Estate Law',
                                'Tax Law (Property-Related)' => 'Tax Law (Property-Related)',
                            ])
                            ->searchable()
                            ->nullable()
                            ->preload(),

                        // Phone input
                        TextInput::make('phone')
                            ->tel()
                            ->nullable()
                            ->maxLength(12),

                        // National ID (nullable, must be unique in advocates table)
                        TextInput::make('national_id')
                            ->nullable()
                            ->maxLength(20)
                            ->unique(
                                table: 'advocates',
                                column: 'national_id',
                                ignorable: fn($record) => $record
                            ),

                        // Address input as a textarea
                        Textarea::make('address')
                            ->label('Address')
                            ->nullable()
                            ->maxLength(255)
                            ->rows(4),
                    ])
                    ->columns(2), // Arrange advocate fields in 2 columns
            ]);
    }

    /**
     * Define the table view for listing advocates.
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // Display name from related user
                TextColumn::make('user.name')->sortable()->searchable(),

                // Display email from related user
                TextColumn::make('user.email')->sortable()->searchable(),

                // Display license number
                TextColumn::make('license_number')->sortable(),

                // Display area of specialization
                TextColumn::make('specialization')->sortable(),

                // Phone number
                TextColumn::make('phone')->searchable(),

                // National ID
                TextColumn::make('national_id')->label('National ID')->searchable(),

                // Address, truncated to 50 characters
                TextColumn::make('address')->searchable()->limit(50),

                // Created at timestamp
                TextColumn::make('created_at')->dateTime(),
            ])
            ->filters([
                // You can add filters here later
            ])
            ->actions([
                // View, edit, and delete individual advocate
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                // Delete multiple advocates at once
                Tables\Actions\DeleteBulkAction::make(),
            ])
            ->recordUrl(fn($record) => AdvocateResource::getUrl('view', ['record' => $record])) // Make row clickable
            ->defaultSort('created_at', 'desc'); // Sort by most recent
    }

    /**
     * Define the pages/routes for this resource.
     */
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAdvocates::route('/'),
            'create' => Pages\CreateAdvocate::route('/create'),
            'view' => Pages\ViewAdvocate::route('/{record}'),
            'edit' => Pages\EditAdvocate::route('/{record}/edit'),
        ];
    }
}
