<?php

namespace App\Filament\Resources\AdminResource\Widgets;
use App\Models\ApartmentDetail;
use App\Models\ProjectExpense;
use App\Models\SubDivisionDetail;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Carbon\Carbon;

class ProjectStats extends BaseWidget
{
    use InteractsWithPageFilters;

    protected static ?int $sort = 2; // After PaymentStats

    protected int|string|array $columnSpan = 'full';

    protected function getStats(): array
    {
        $filters = $this->filters ?? [];
        $startDate = isset($filters['startDate']) && $filters['startDate'] ? Carbon::parse($filters['startDate']) : null;
        $endDate = isset($filters['endDate']) && $filters['endDate'] ? Carbon::parse($filters['endDate']) : null;

        // Total Revenue (from amount_charged in apartment_details and subdivision_details)
        $apartmentRevenueQuery = ApartmentDetail::query();
        $subdivisionRevenueQuery = SubDivisionDetail::query();
        $expenseQuery = ProjectExpense::query()->join('expenses', 'project_expenses.expense_id', '=', 'expenses.id');

        if ($startDate) {
            $apartmentRevenueQuery->where('created_at', '>=', $startDate);
            $subdivisionRevenueQuery->where('created_at', '>=', $startDate);
            $expenseQuery->where('expenses.created_at', '>=', $startDate); // Changed to created_at
        }

        if ($endDate) {
            $apartmentRevenueQuery->where('created_at', '<=', $endDate);
            $subdivisionRevenueQuery->where('created_at', '<=', $endDate);
            $expenseQuery->where('expenses.created_at', '<=', $endDate); // Changed to created_at
        }

        $totalApartmentRevenue = $apartmentRevenueQuery->sum('amount_charged');
        $totalSubdivisionRevenue = $subdivisionRevenueQuery->sum('amount_charged');
        $totalRevenue = $totalApartmentRevenue + $totalSubdivisionRevenue;
        $totalExpenses = $expenseQuery->sum('expenses.amount_paid'); // Use amount_paid from expenses table

        // Previous period for comparison
        $prevStartDate = $startDate ? $startDate->copy()->subMonth() : now()->subMonth()->startOfMonth();
        $prevEndDate = $endDate ? $endDate->copy()->subMonth() : now()->subMonth()->endOfMonth();

        $prevApartmentRevenue = ApartmentDetail::whereBetween('created_at', [$prevStartDate, $prevEndDate])->sum('amount_charged');
        $prevSubdivisionRevenue = SubDivisionDetail::whereBetween('created_at', [$prevStartDate, $prevEndDate])->sum('amount_charged');
        $prevTotalRevenue = $prevApartmentRevenue + $prevSubdivisionRevenue;
        $prevTotalExpenses = ProjectExpense::query()
            ->join('expenses', 'project_expenses.expense_id', '=', 'expenses.id')
            ->whereBetween('expenses.created_at', [$prevStartDate, $prevEndDate]) // Changed to created_at
            ->sum('expenses.amount_paid'); // Use amount_paid from expenses table

        $profit = $totalRevenue - $totalExpenses;
        $prevProfit = $prevTotalRevenue - $prevTotalExpenses;

        $profitChange = $prevProfit != 0
            ? (($profit - $prevProfit) / abs($prevProfit)) * 100
            : ($profit > 0 ? 100 : ($profit < 0 ? -100 : 0));

        // Monthly chart data (revenue vs expenses)
        $revenueChart = $this->getMonthlyTotals('revenue', $startDate, $endDate);
        $expenseChart = $this->getMonthlyTotals('expense', $startDate, $endDate);

        return [
            Stat::make('Total Project Revenue', 'KES ' . number_format($totalRevenue, 2))
                ->description('From apartments and subdivisions (amount charged)')
                ->color('success')
                ->chart($revenueChart),
            Stat::make('Total Project Expenses', 'KES ' . number_format($totalExpenses, 2))
                ->description('From project expenses (amount paid)')
                ->color('danger')
                ->chart($expenseChart),
            Stat::make('Profit/Loss', 'KES ' . number_format($profit, 2))
                ->description(sprintf('%.2f%% %s vs last period', abs($profitChange), $profitChange >= 0 ? 'increase' : 'decrease'))
                ->descriptionIcon($profitChange >= 0 ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down')
                ->color($profit >= 0 ? 'success' : 'danger'),
        ];
    }

    protected function getMonthlyTotals(string $type, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $totals = [];
        $start = $startDate ? $startDate->copy()->startOfMonth() : now()->subMonths(5)->startOfMonth();
        $end = $endDate ? $endDate->copy()->endOfMonth() : now()->endOfMonth();

        while ($start->lte($end)) {
            if ($type === 'revenue') {
                $apartmentTotal = ApartmentDetail::whereYear('created_at', $start->year)
                    ->whereMonth('created_at', $start->month)
                    ->sum('amount_charged');
                $subdivisionTotal = SubDivisionDetail::whereYear('created_at', $start->year)
                    ->whereMonth('created_at', $start->month)
                    ->sum('amount_charged');
                $monthTotal = $apartmentTotal + $subdivisionTotal;
            } else {
                $monthTotal = ProjectExpense::query()
                    ->join('expenses', 'project_expenses.expense_id', '=', 'expenses.id')
                    ->whereYear('expenses.created_at', $start->year) // Changed to created_at
                    ->whereMonth('expenses.created_at', $start->month) // Changed to created_at
                    ->sum('expenses.amount_paid'); // Use amount_paid from expenses table
            }
            $totals[] = (float) $monthTotal; // Cast to float for decimal support
            $start->addMonth();
        }

        return $totals;
    }

    public static function canView(): bool
    {
        $user = auth()->user();

        return $user && ($user->hasRole('admin') || $user->hasRole('super_admin'));
    }
}
