<?php

namespace App\Filament\Resources\AdminResource\Widgets;
use App\Models\Payment;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Carbon\Carbon;

class PaymentStats extends BaseWidget
{
    use InteractsWithPageFilters;

    protected static ?int $sort = 1;

    protected int|string|array $columnSpan = 'full';

    protected function getStats(): array
    {
        // Safely access filters with fallback
        $filters = $this->filters ?? [];
        $startDate = isset($filters['startDate']) && $filters['startDate'] ? Carbon::parse($filters['startDate']) : null;
        $endDate = isset($filters['endDate']) && $filters['endDate'] ? Carbon::parse($filters['endDate']) : null;

        // Apply date filters
        $outgoingQuery = Payment::where('direction', 'outgoing');
        $incomingQuery = Payment::where('direction', 'incoming');
        $salaryQuery = Payment::where('payable_type', 'App\Models\Salary');
        $officeUtilityQuery = Payment::where('payable_type', 'App\Models\OfficeUtility');
        $projectQuery = Payment::where('payable_type', 'App\Models\Project');
        $conveyancingQuery = Payment::where('payable_type', 'App\Models\Conveyancing');
        $litigationQuery = Payment::where('payable_type', 'App\Models\Litigation');

        if ($startDate) {
            $outgoingQuery->where('created_at', '>=', $startDate);
            $incomingQuery->where('created_at', '>=', $startDate);
            $salaryQuery->where('created_at', '>=', $startDate);
            $officeUtilityQuery->where('created_at', '>=', $startDate);
            $projectQuery->where('created_at', '>=', $startDate);
            $conveyancingQuery->where('created_at', '>=', $startDate);
            $litigationQuery->where('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $outgoingQuery->where('created_at', '<=', $endDate);
            $incomingQuery->where('created_at', '<=', $endDate);
            $salaryQuery->where('created_at', '<=', $endDate);
            $officeUtilityQuery->where('created_at', '<=', $endDate);
            $projectQuery->where('created_at', '<=', $endDate);
            $conveyancingQuery->where('created_at', '<=', $endDate);
            $litigationQuery->where('created_at', '<=', $endDate);
        }

        // Totals for current period
        $totalOutgoing = $outgoingQuery->sum('amount_paid');
        $totalIncoming = $incomingQuery->sum('amount_paid');
        $totalSalary = $salaryQuery->sum('amount_paid');
        $totalOfficeUtility = $officeUtilityQuery->sum('amount_paid');
        $totalProject = $projectQuery->sum('amount_paid');
        $totalConveyancing = $conveyancingQuery->sum('amount_paid');
        $totalLitigation = $litigationQuery->sum('amount_paid');

        // Previous period for trends
        $prevStartDate = $startDate ? $startDate->copy()->subMonth() : now()->subMonth()->startOfMonth();
        $prevEndDate = $endDate ? $endDate->copy()->subMonth() : now()->subMonth()->endOfMonth();

        $totalOutgoingPrevious = Payment::where('direction', 'outgoing')
            ->whereBetween('created_at', [$prevStartDate, $prevEndDate])
            ->sum('amount_paid');
        $totalIncomingPrevious = Payment::where('direction', 'incoming')
            ->whereBetween('created_at', [$prevStartDate, $prevEndDate])
            ->sum('amount_paid');

        $outgoingChange = $totalOutgoingPrevious > 0
            ? (($totalOutgoing - $totalOutgoingPrevious) / $totalOutgoingPrevious) * 100
            : ($totalOutgoing > 0 ? 100 : 0);
        $incomingChange = $totalIncomingPrevious > 0
            ? (($totalIncoming - $totalIncomingPrevious) / $totalIncomingPrevious) * 100
            : ($totalIncoming > 0 ? 100 : 0);

        // Monthly chart data
        $outgoingChart = $this->getMonthlyTotals('outgoing', $startDate, $endDate);
        $incomingChart = $this->getMonthlyTotals('incoming', $startDate, $endDate);
        $salaryChart = $this->getMonthlyTotalsByType('App\Models\Salary', $startDate, $endDate);
        $officeUtilityChart = $this->getMonthlyTotalsByType('App\Models\OfficeUtility', $startDate, $endDate);
        $projectChart = $this->getMonthlyTotalsByType('App\Models\Project', $startDate, $endDate);
        $conveyancingChart = $this->getMonthlyTotalsByType('App\Models\Conveyancing', $startDate, $endDate);
        $litigationChart = $this->getMonthlyTotalsByType('App\Models\Litigation', $startDate, $endDate);

        return [
            Stat::make('Total Outgoing Payments', 'KES ' . number_format($totalOutgoing, 0))
                ->description(sprintf('%.2f%% %s vs last period', abs($outgoingChange), $outgoingChange >= 0 ? 'increase' : 'decrease'))
                ->descriptionIcon($outgoingChange >= 0 ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down')
                ->color($outgoingChange >= 0 ? 'success' : 'danger')
                ->chart($outgoingChart),
            Stat::make('Total Incoming Payments', 'KES ' . number_format($totalIncoming, 0))
                ->description(sprintf('%.2f%% %s vs last period', abs($incomingChange), $incomingChange >= 0 ? 'increase' : 'decrease'))
                ->descriptionIcon($incomingChange >= 0 ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down')
                ->color($incomingChange >= 0 ? 'success' : 'danger')
                ->chart($incomingChart),
            Stat::make('Total Salary Payments', 'KES ' . number_format($totalSalary, 0))
                ->color('primary')
                ->chart($salaryChart),
            Stat::make('Total Office Utility Payments', 'KES ' . number_format($totalOfficeUtility, 0))
                ->color('primary')
                ->chart($officeUtilityChart),
            Stat::make('Total Project Payments', 'KES ' . number_format($totalProject, 0))
                ->color('primary')
                ->chart($projectChart),
            Stat::make('Total Conveyancing Payments', 'KES ' . number_format($totalConveyancing, 0))
                ->color('primary')
                ->chart($conveyancingChart),
            Stat::make('Total Litigation Payments', 'KES ' . number_format($totalLitigation, 0))
                ->color('primary')
                ->chart($litigationChart),
        ];
    }

    protected function getMonthlyTotals(string $direction, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $totals = [];
        $start = $startDate ? $startDate->copy()->startOfMonth() : now()->subMonths(5)->startOfMonth();
        $end = $endDate ? $endDate->copy()->endOfMonth() : now()->endOfMonth();

        while ($start->lte($end)) {
            $monthTotal = Payment::where('direction', $direction)
                ->whereYear('created_at', $start->year)
                ->whereMonth('created_at', $start->month)
                ->sum('amount_paid');
            $totals[] = (int) $monthTotal;
            $start->addMonth();
        }

        return $totals;
    }

    protected function getMonthlyTotalsByType(string $payableType, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $totals = [];
        $start = $startDate ? $startDate->copy()->startOfMonth() : now()->subMonths(5)->startOfMonth();
        $end = $endDate ? $endDate->copy()->endOfMonth() : now()->endOfMonth();

        while ($start->lte($end)) {
            $monthTotal = Payment::where('payable_type', $payableType)
                ->whereYear('created_at', $start->year)
                ->whereMonth('created_at', $start->month)
                ->sum('amount_paid');
            $totals[] = (int) $monthTotal;
            $start->addMonth();
        }

        return $totals;
    }

    //controlling access
    public static function canView(): bool
{
    $user = auth()->user();

    return $user && ($user->hasRole('admin') || $user->hasRole('super_admin'));
}
}